#!/bin/bash
#
# Copyright 2024 Amazon.com, Inc. or its affiliates. All rights reserved.
#
# AMAZON PROPRIETARY/CONFIDENTIAL
#
# You may not use this file except in compliance with the terms and conditions
# set forth in the accompanying LICENSE.TXT file. This file is a
# Modifiable File, as defined in the accompanying LICENSE.TXT file.
#
# THESE MATERIALS ARE PROVIDED ON AN "AS IS" BASIS. AMAZON SPECIFICALLY
# DISCLAIMS, WITH RESPECT TO THESE MATERIALS, ALL WARRANTIES, EXPRESS,
# IMPLIED, OR STATUTORY, INCLUDING THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.
#

# This script enables firewall settings for the device
IPTABLES=/usr/sbin/iptables
HOSTAP_IF=ap0
HOSTAP_IP=10.1.5.1

# Firewall settings for azt hostap interface
azt_provision_firewall_start () {
    if [ "$#" -ne 1 ]; then
        echo "azt provision enable: Using default parameters"
    else
        HOSTAP_IF=$1
	HOSTAP_IP=$2
    fi
    echo "Setting up azt provision hostap firewall settings on $HOSTAP_IF"

    # Setup ip tables to reject all non-essential traffic
    # Redirect all DNS traffic to ourselves. This is necessary when the client device
    # has a static DNS address configured
    $IPTABLES -t nat -A PREROUTING -i "$HOSTAP_IF" -p udp --dport 53 -j DNAT --to ${HOSTAP_IP}
    # ACCEPT all DNS traffic
    $IPTABLES -A INPUT -i "$HOSTAP_IF" -p udp --dport 53 -j ACCEPT
    # ACCEPT all DHCP traffic
    $IPTABLES -A INPUT -i "$HOSTAP_IF" -p udp --dport 67:68 --sport 67:68 -j ACCEPT
    # ACCEPT incoming azt provisionee udp traffic
    $IPTABLES -A INPUT -i "$HOSTAP_IF" -p udp --dport 8888 -j ACCEPT
    # ACCEPT all incoming azt webserver traffic
    $IPTABLES -A INPUT -i "$HOSTAP_IF" -p tcp --dport 8080 -j ACCEPT

    # Allow all outgoing traffic
    $IPTABLES -A OUTPUT -o "$HOSTAP_IF" -j ACCEPT
    $IPTABLES -A FORWARD -i "$HOSTAP_IF" -j DROP
    $IPTABLES -A INPUT -i "$HOSTAP_IF" -j DROP
}

# Firewall settings for azt provision hostap interface
azt_provision_firewall_stop () {
    if [ "$#" -ne 1 ]; then
        echo "azt provision disable: Using default parameters"
    else
        HOSTAP_IF=$1
    fi

    echo "Disabling azt provision hostap firewall settings on $HOSTAP_IF"

    # Delete all rules setup by azt_provision_firewall_start
    # Delete DNS redirection
    $IPTABLES -t nat -D PREROUTING -i "$HOSTAP_IF" -p udp --dport 53 -j DNAT --to ${HOSTAP_IP}
    # Delete ACCEPT all DNS traffic
    $IPTABLES -D INPUT -i "$HOSTAP_IF" -p udp --dport 53 -j ACCEPT
    # Delete ACCEPT all DHCP traffic
    $IPTABLES -D INPUT -i "$HOSTAP_IF" -p udp --dport 67:68 --sport 67:68 -j ACCEPT
    # Delete incoming azt provisionee udp traffic
    $IPTABLES -D INPUT -i "$HOSTAP_IF" -p udp --dport 8888 -j ACCEPT
    # Delete all incoming azt webserver traffic
    $IPTABLES -D INPUT -i "$HOSTAP_IF" -p tcp --dport 8080 -j ACCEPT
    # Delete allow all outgoing traffic
    $IPTABLES -D OUTPUT -o "$HOSTAP_IF" -j ACCEPT
    $IPTABLES -D FORWARD -i "$HOSTAP_IF" -j DROP
    $IPTABLES -D INPUT -i "$HOSTAP_IF" -j DROP
}

start_setup_firewall () {

    case "$1" in
        azt_provision)
            shift
            azt_provision_firewall_start "$@"
            ;;
        *)
            exit 1
            ;;
    esac
}

stop_setup_firewall () {

    case "$1" in
        azt_provision)
            shift
            azt_provision_firewall_stop "$@"
            ;;
        *)
            exit 1
            ;;
    esac
}

case "$1" in
    start)
        if [ "$2" ]
        then
            shift
            # Setup any specific settings
            start_setup_firewall "$@"
            exit 0
        fi
        ;;
    stop)
        if [ "$2" ]
        then
            shift
            # Stop any specific settings
            stop_setup_firewall "$@"
            exit 0
        fi
        ;;
    *)
        exit 1
        ;;
esac

exit 1

