// Global scanner state manager to prevent multiple scanner instances
class ScannerManager {
    private static instance: ScannerManager;
    private activeScanners: Set<string> = new Set();
    private lastDetection: { content: string; timestamp: number } | null = null;
    private readonly DETECTION_COOLDOWN = 5000; // 5 seconds

    private constructor() {}

    static getInstance(): ScannerManager {
        if (!ScannerManager.instance) {
            ScannerManager.instance = new ScannerManager();
        }
        return ScannerManager.instance;
    }

    registerScanner(scannerId: string): boolean {
        console.log(`📝 Registering scanner: ${scannerId}`);
        
        // If there's already an active scanner, reject this one
        if (this.activeScanners.size > 0) {
            console.warn(`⚠️ Scanner ${scannerId} rejected - another scanner is active:`, Array.from(this.activeScanners));
            return false;
        }

        this.activeScanners.add(scannerId);
        console.log(`✅ Scanner ${scannerId} registered successfully`);
        return true;
    }

    unregisterScanner(scannerId: string): void {
        console.log(`📝 Unregistering scanner: ${scannerId}`);
        this.activeScanners.delete(scannerId);
        console.log(`✅ Scanner ${scannerId} unregistered. Active scanners:`, Array.from(this.activeScanners));
    }

    shouldProcessDetection(scannerId: string, content: string): boolean {
        const now = Date.now();

        // Check if this scanner is registered
        if (!this.activeScanners.has(scannerId)) {
            console.warn(`⚠️ Scanner ${scannerId} not registered, ignoring detection`);
            return false;
        }

        // Check for duplicate detection within cooldown period
        if (this.lastDetection && 
            this.lastDetection.content === content && 
            (now - this.lastDetection.timestamp) < this.DETECTION_COOLDOWN) {
            console.log(`🔄 Duplicate detection ignored (within ${this.DETECTION_COOLDOWN}ms cooldown)`);
            return false;
        }

        // Update last detection
        this.lastDetection = { content, timestamp: now };
        console.log(`✅ Detection approved for scanner ${scannerId}: ${content}`);
        return true;
    }

    getActiveScannersCount(): number {
        return this.activeScanners.size;
    }

    getActiveScanners(): string[] {
        return Array.from(this.activeScanners);
    }

    clearLastDetection(): void {
        console.log('🧹 Clearing last detection from global state');
        this.lastDetection = null;
    }

    forceUnregisterAll(): void {
        console.log('🚨 Force unregistering all scanners');
        this.activeScanners.clear();
        this.lastDetection = null;
    }
}

export const scannerManager = ScannerManager.getInstance();
