// src/utils/envValidation.ts

export interface EnvironmentConfig {
    region: string;
    endpoint: string;
    accessKeyId: string;
    secretAccessKey: string;
    sessionToken: string;
}

export function validateEnvironmentConfig(): EnvironmentConfig {
    const requiredVars = [
        'REACT_APP_AWS_REGION',
        'REACT_APP_IOT_ENDPOINT',
        'REACT_APP_AWS_ACCESS_KEY_ID',
        'REACT_APP_AWS_SECRET_ACCESS_KEY',
        'REACT_APP_AWS_SESSION_TOKEN'
    ];

    const missing = requiredVars.filter(varName => !process.env[varName]);
    
    if (missing.length > 0) {
        throw new Error(`Missing required environment variables: ${missing.join(', ')}`);
    }

    const endpoint = process.env.REACT_APP_IOT_ENDPOINT!;
    
    // Validate endpoint URL format
    if (!endpoint.startsWith('https://') && !endpoint.startsWith('http://')) {
        throw new Error(`Invalid endpoint URL: ${endpoint}. Must start with https:// or http://`);
    }

    try {
        new URL(endpoint);
    } catch (error) {
        throw new Error(`Invalid endpoint URL format: ${endpoint}. Error: ${error instanceof Error ? error.message : 'Unknown error'}`);
    }

    return {
        region: process.env.REACT_APP_AWS_REGION!,
        endpoint: endpoint,
        accessKeyId: process.env.REACT_APP_AWS_ACCESS_KEY_ID!,
        secretAccessKey: process.env.REACT_APP_AWS_SECRET_ACCESS_KEY!,
        sessionToken: process.env.REACT_APP_AWS_SESSION_TOKEN!
    };
}

export function logEnvironmentInfo(): void {
    try {
        const config = validateEnvironmentConfig();
        console.log('Environment configuration:', {
            region: config.region,
            endpoint: config.endpoint,
            hasAccessKey: !!config.accessKeyId,
            hasSecretKey: !!config.secretAccessKey,
            hasSessionToken: !!config.sessionToken,
            accessKeyPrefix: config.accessKeyId.substring(0, 8) + '...',
            sessionTokenPrefix: config.sessionToken.substring(0, 20) + '...'
        });
    } catch (error) {
        console.error('Environment validation failed:', error);
    }
}
