// src/types/deviceTypes.ts

export enum ProtocolType {
    ZIGBEE = 'ZIGBEE',
    ZWAVE = 'ZWAVE'
}

// Device type constants
export const MATTER_DEVICE_TYPES = {
    ON_OFF_LIGHT: 'OnOffLight',
    DIMMABLE_LIGHT: 'DimmableLight',
    COLOR_LIGHT: 'ColorLight',
    ON_OFF_SWITCH: 'OnOffSwitch',
    TEMPERATURE_SENSOR: 'Temperature Sensor',
    CONTACT_SENSOR: 'Contact Sensor',
    OCCUPANCY_SENSOR: 'Occupancy Sensor',
    DOOR_LOCK: 'Door Lock',
    WINDOW_COVERING: 'Window Covering',
    THERMOSTAT: 'Thermostat',
    UNKNOWN: 'Unknown Device',
} as const;

// Helper type for device capabilities
export const MATTER_CAPABILITIES = {
    ONOFF: 'OnOff',
    LEVEL_CONTROL: 'LevelControl',
    COLOR_CONTROL: 'ColorControl',
    TEMPERATURE_MEASUREMENT: 'matter.TemperatureMeasurement@1.4',
    OCCUPANCY_SENSING: 'matter.OccupancySensing@1.4',
    DOOR_LOCK: 'matter.DoorLock@1.4',
    WINDOW_COVERING: 'matter.WindowCovering@1.4',
} as const;

// Base types for identifiers and versioning
type MatterVersion = '1.4';

// Property types
export interface PropertyValue {
    propertyValue: any;
    lastChangedAt?: string;
}

export interface PropertyMetadata {
    name: string;
    value?: PropertyValue;
}

export interface PropertyDefinition {
    name: string;
    type: string;
    access: string[];
    dataType?: 'boolean' | 'number' | 'string' | 'object';
    readable?: boolean;
    writable?: boolean;
}

// Capability types
export interface Capability {
    id: string;           // e.g., "matter.OnOff@1.4"
    name: string;         // e.g., "On/Off"
    version: string;      // e.g., "6" or "1.4"
    properties: PropertyMetadata[];
    actions: string[];    // e.g., ["Off", "On", "Toggle"]
    events: string[];
}

export interface CapabilityDefinition {
    id: string;
    name: string;
    version: string;
    properties: PropertyDefinition[];
    actions: string[];
    events: string[];
}

// Endpoint types
export interface Endpoint {
    endpointId: string;
    deviceTypes: string[];
    capabilities: Capability[];
}

export interface EndpointDefinition {
    id: string;
    deviceTypes: string[];
    capabilities: CapabilityDefinition[];
}

// Device state and capability report types
export interface DeviceState {
    Endpoints: Endpoint[];
}

export interface CapabilityReport {
    version: string;
    nodeId: string;
    endpoints: EndpointDefinition[];
}

// Main device type
export interface Device {
    ManagedThingId: string;
    Name: string;
    Protocol: ProtocolType;
    CapabilityReport: CapabilityReport;
    currentState?: DeviceState;
}

// Specific capability types
export interface OnOffCapability extends Capability {
    id: string;
    name: 'OnOff';
    version: string;
    properties: PropertyMetadata[];
    actions: string[];
    events: string[];
}

// Helper functions and types
export const isOnOffCapability = (capability: Capability): capability is OnOffCapability => {
    return capability.id.includes(MATTER_CAPABILITIES.ONOFF);
};

export type DeviceType = typeof MATTER_DEVICE_TYPES[keyof typeof MATTER_DEVICE_TYPES];
export type CapabilityType = typeof MATTER_CAPABILITIES[keyof typeof MATTER_CAPABILITIES];
