import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { notificationService } from '../services/notificationService';
import { DeviceState } from '../types/deviceTypes';
import { updateDeviceState } from './deviceSlice';

interface SubscriptionState {
    subscribedDevices: string[];
}

const initialState: SubscriptionState = {
    subscribedDevices: []
};

const subscriptionSlice = createSlice({
    name: 'subscriptions',
    initialState,
    reducers: {
        subscribeToDevice: (state, action: PayloadAction<string>) => {
            const deviceId = action.payload;
            if (!state.subscribedDevices.includes(deviceId)) {
                state.subscribedDevices.push(deviceId);
            }
        },
        unsubscribeFromDevice: (state, action: PayloadAction<string>) => {
            const deviceId = action.payload;
            state.subscribedDevices = state.subscribedDevices.filter(id => id !== deviceId);
        }
    }
});

export const { subscribeToDevice, unsubscribeFromDevice } = subscriptionSlice.actions;

// Create a middleware to handle subscriptions
export const subscriptionMiddleware = (store: any) => (next: any) => (action: any) => {
    if (subscribeToDevice.match(action)) {
        const deviceId = action.payload;
        console.log('Setting up device updates subscription for device:', deviceId);
        notificationService.subscribeToDeviceUpdates(deviceId, (newState: DeviceState) => {
            store.dispatch(updateDeviceState({ ManagedThingId: deviceId, newState }));
        });
    }
    return next(action);
};

export default subscriptionSlice.reducer; 