// src/store/hubSlice.ts
import { createSlice, createAsyncThunk } from '@reduxjs/toolkit';
import { ListManagedThingsCommand } from '@aws-sdk/client-iot-managed-integrations';
import { deviceService } from '../services/deviceService'; // Import your existing service
import { deviceStatePollingService } from '../services/deviceStatePollingService';
import { HubState } from '../types/hubTypes';
import {RootState} from "./store";
import {addDevice, updateDeviceState, removeDevice} from "./deviceSlice";
import { HUB_CONFIG } from '../constants';




const initialState: HubState = {
    hubId: HUB_CONFIG.MANAGED_THING_ID,
    devices: [],
    loading: false,
    error: null
};

export const fetchAndAddAllDevices = createAsyncThunk(
    'devices/fetchAndAddAll',
    async (_, { dispatch, getState }) => {
        try {
            // First fetch the list of managed things
            const managedThings = await fetchManagedThings();
            console.log('Fetched managed things:', managedThings);

            // Get current state
            const state = getState() as RootState;
            const existingDevices = state.devices.devices;

            // Create a map of existing devices for quick lookup
            const existingDeviceMap = new Map(
                existingDevices.map(device => [device.ManagedThingId, device])
            );

            // Process each device
            for (const thingId of managedThings) {
                try {
                    // Check if device exists and has a name
                    const existingDevice = existingDeviceMap.get(thingId);
                    
                    // Fetch latest device details
                    const deviceDetails = await deviceService.getDeviceWithDetails(thingId);

                    // Preserve existing name if it exists
                    if (existingDevice?.Name && existingDevice.Name !== 'Unnamed Device') {
                        console.log(`Preserving existing name for device ${thingId}: "${existingDevice.Name}" (was "${deviceDetails.Name}")`);
                        deviceDetails.Name = existingDevice.Name;
                    } else {
                        console.log(`No existing name to preserve for device ${thingId}. Existing: "${existingDevice?.Name}", New: "${deviceDetails.Name}"`);
                    }

                    // Add or update device in store
                    dispatch(addDevice(deviceDetails));
                    
                    // Don't automatically start polling - let page-based control handle it
                    console.log(`Device ${thingId} added to store - polling controlled by page mode`);

                } catch (error: any) {
                    console.error(`Error processing device ${thingId}:`, error);
                    // If device not found, remove it from store
                    if (error?.message?.includes('404') || error?.name === 'ResourceNotFoundException') {
                        console.log(`Removing non-existent device ${thingId} from store`);
                        dispatch(removeDevice(thingId));
                    }
                }
            }

            return managedThings;
        } catch (error: any) {
            console.error('Error fetching managed things:', error);
            throw new Error(error?.message || 'Failed to fetch managed things');
        }
    }
);


async function fetchManagedThings(): Promise<string[]> {
        console.log('🚀 Starting fetchManagedThings...');
        try {
            const command = new ListManagedThingsCommand({
                ParentControllerIdentifierFilter: initialState.hubId
            });
            console.log('📤 Sending request with command:', command);

            const response = await deviceService.client.send(command);
            console.log('📥 Received response:', response);

            const filteredDevices = response.Items
                ?.filter(thing => {
                    console.log('Checking thing:', thing);
                    return thing.Role === "DEVICE" &&
                        thing.ProvisioningStatus === "ACTIVATED";
                })
                .map(thing => thing.Id || "");

            console.log('✅ Filtered devices:', filteredDevices);
            return filteredDevices || [];
        } catch (error: any) {
            console.error('❌ Error in fetchManagedThings:', error);
            return [];
        }
    }

export const fetchAndHub = createAsyncThunk(
    'devices/fetchAndAddAll',
    async (_, { dispatch }) => {
        try {
            // Await the fetchManagedThings call
            return await fetchManagedThings();
        } catch (error) {
            console.error('Error fetching managed things:', error);
            throw error;
        }
    }
);

const hubSlice = createSlice({
    name: 'hub',
    initialState,
    reducers: {},
    extraReducers: (builder) => {
        builder
            .addCase(fetchAndHub.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(fetchAndHub.fulfilled, (state, action) => {
                state.loading = false;
                // Filter and map to get device IDs
                state.devices = action.payload
            })
            .addCase(fetchAndHub.rejected, (state, action) => {
                state.loading = false;
                state.error = action.payload as string;
            });
    }
});

export default hubSlice.reducer;

// Selectors

export const selectManagedThings = (state: RootState) => state.hub.devices;
export const selectHubLoading = (state: RootState) => state.hub.loading;
export const selectHubError = (state: RootState) => state.hub.error;
