// src/store/deviceSlice.ts

import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { Device, DeviceState as DeviceStateType } from '../types/deviceTypes';
import {RootState} from "./store";

interface DevicesState {
    devices: Device[];
    loading: boolean;
    error: string | null;
}

const initialState: DevicesState = {
    devices: [],
    loading: false,
    error: null
};

const deviceSlice = createSlice({
    name: 'devices',
    initialState,
    reducers: {
        addDevice: (state, action: PayloadAction<Device>) => {
            // Check if device already exists
            const existingDeviceIndex = state.devices.findIndex(
                device => device.ManagedThingId === action.payload.ManagedThingId
            );

            if (existingDeviceIndex >= 0) {
                // Update existing device
                const existingDevice = state.devices[existingDeviceIndex];
                const preservedName = existingDevice.Name !== 'Unnamed Device' ? existingDevice.Name : action.payload.Name;
                
                console.log(`addDevice: Updating existing device ${action.payload.ManagedThingId}. Existing name: "${existingDevice.Name}", New name: "${action.payload.Name}", Preserved name: "${preservedName}"`);
                
                state.devices[existingDeviceIndex] = {
                    ...action.payload,
                    // Preserve the name if it exists and isn't the default
                    Name: preservedName,
                    // Preserve the current state if the new state is undefined
                    currentState: action.payload.currentState || existingDevice.currentState
                };
            } else {
                // Add new device
                console.log(`addDevice: Adding new device ${action.payload.ManagedThingId} with name: "${action.payload.Name}"`);
                state.devices.push(action.payload);
            }
        },
        updateDeviceState: (state, action: PayloadAction<{
            ManagedThingId: string;
            newState: DeviceStateType
        }>) => {
            const { ManagedThingId, newState } = action.payload;
            const deviceIndex = state.devices.findIndex(
                device => device.ManagedThingId === ManagedThingId
            );

            if (deviceIndex >= 0) {
                // Update state while preserving other device properties
                state.devices[deviceIndex] = {
                    ...state.devices[deviceIndex],
                    currentState: newState
                };
            }
        },
        updateDeviceName: (state, action: PayloadAction<{
            ManagedThingId: string;
            name: string;
        }>) => {
            const { ManagedThingId, name } = action.payload;
            const deviceIndex = state.devices.findIndex(
                device => device.ManagedThingId === ManagedThingId
            );

            if (deviceIndex >= 0) {
                const oldName = state.devices[deviceIndex].Name;
                state.devices[deviceIndex].Name = name;
                console.log(`updateDeviceName: Updated device ${ManagedThingId} name from "${oldName}" to "${name}"`);
            } else {
                console.log(`updateDeviceName: Device ${ManagedThingId} not found in store`);
            }
        },
        removeDevice: (state, action: PayloadAction<string>) => {
            state.devices = state.devices.filter(
                device => device.ManagedThingId !== action.payload
            );
        },
        resetDevices: (state) => {
            // Reset to initial state
            state.devices = [];
            state.loading = false;
            state.error = null;
        },
        setLoading: (state, action: PayloadAction<boolean>) => {
            state.loading = action.payload;
        },
        setError: (state, action: PayloadAction<string | null>) => {
            state.error = action.payload;
        }
    }
});

export const selectDeviceById = (state: RootState, deviceId: string) =>
    state.devices.devices.find(device => device.ManagedThingId === deviceId);

export const {
    addDevice,
    updateDeviceState,
    updateDeviceName,
    removeDevice,
    resetDevices,
    setLoading,
    setError
} = deviceSlice.actions;

export default deviceSlice.reducer;
