// src/services/notificationService.ts

import {
    KinesisClient,
    GetRecordsCommand,
    GetShardIteratorCommand,
    DescribeStreamCommand,
    _Record,
    SubscribeToShardCommand,
    SubscribeToShardCommandInput,
    SubscribeToShardCommandOutput
} from '@aws-sdk/client-kinesis';
import { 
    SendManagedThingCommandCommand,
    CommandEndpoint,
    CommandCapability
} from '@aws-sdk/client-iot-managed-integrations';
import { fromEnv } from '@aws-sdk/credential-provider-env';
import { DeviceState } from '../types/deviceTypes';
import { deviceService } from './deviceService';
import { store } from '../store/store';
import { updateDeviceState } from '../store/deviceSlice';

interface DeviceStateMessage {
    messageType: 'DEVICE_STATE';
    timestamp: string;
    payload: {
        addedStates?: {
            endpoints: any[];
        };
        modifiedStates?: {
            endpoints: any[];
        };
    };
    resources: string[];
}

interface DeviceEventMessage {
    messageType: 'DEVICE_EVENT';
    timestamp: string;
    payload: {
        endpoints: Array<{
            endpointId: string;
            capabilities: Array<{
                id: string;
                name: string;
                version: string;
                properties: Array<{
                    name: string;
                    value: any;
                }>;
            }>;
        }>;
    };
    resources: string[];
}

interface DeviceLifecycleMessage {
    messageType: 'DEVICE_LIFE_CYCLE';
    timestamp: string;
    payload: {
        deviceDetails: {
            id: string;
            arn: string;
            createdAt: string;
            updatedAt: string;
        };
        status: 'DISCOVERED' | 'DELETED' | 'UPDATED';
    };
    resources: string[];
}

type MessageHandler<T> = (deviceId: string, message: T) => void;

export class NotificationService {
    private static instance: NotificationService | null = null;
    private kinesisClient: KinesisClient;
    private streams: {
        state: string;
        event: string;
        lifecycle: string;
    };
    private subscribers: Map<string, Set<(state: DeviceState) => void>>;
    private isPolling: boolean = false;
    private pollingTimeout: NodeJS.Timeout | null = null;
    private backoffInterval: number = 1000;
    private maxBackoffInterval: number = 30000;
    private lastProcessedTimestamp: number = 0;
    private processedRecords: Set<string> = new Set();
    private readonly MAX_PROCESSED_RECORDS = 1000; // Keep track of last 1000 processed records
    private isInitialized: boolean = false;
    private initializationPromise: Promise<void> | null = null;
    private readonly BASE_POLLING_INTERVAL = 5000; // 5 seconds base interval
    private readonly MAX_JITTER = 1000; // 1 second max jitter
    private currentBackoff = this.BASE_POLLING_INTERVAL;
    private pollingSetupPromise: Promise<void> | null = null; // Track polling setup promise
    private lastStateRefreshTime: number = 0; // Track when we last refreshed device states

    private constructor() {
        // Debug log environment variables
        console.log('AWS Environment Variables:', {
            region: process.env.REACT_APP_AWS_REGION,
            hasAccessKeyId: !!process.env.REACT_APP_AWS_ACCESS_KEY_ID,
            hasSecretKey: !!process.env.REACT_APP_AWS_SECRET_ACCESS_KEY,
            hasSessionToken: !!process.env.REACT_APP_AWS_SESSION_TOKEN,
            streamName: 'IoTmi-device-event-stream',
            shardId: 'shardId-000000000002'
        });

        // Validate required environment variables
        if (!process.env.REACT_APP_AWS_REGION) {
            throw new Error('AWS Region is not configured');
        }
        if (!process.env.REACT_APP_AWS_ACCESS_KEY_ID) {
            throw new Error('AWS Access Key ID is not configured');
        }
        if (!process.env.REACT_APP_AWS_SECRET_ACCESS_KEY) {
            throw new Error('AWS Secret Access Key is not configured');
        }

        this.kinesisClient = new KinesisClient({
            region: process.env.REACT_APP_AWS_REGION,
            credentials: {
                accessKeyId: process.env.REACT_APP_AWS_ACCESS_KEY_ID,
                secretAccessKey: process.env.REACT_APP_AWS_SECRET_ACCESS_KEY,
                ...(process.env.REACT_APP_AWS_SESSION_TOKEN && {
                    sessionToken: process.env.REACT_APP_AWS_SESSION_TOKEN
                })
            },
            // Add explicit retry configuration
            maxAttempts: 1, // Disable SDK's automatic retries
            requestHandler: {
                abortSignal: undefined,
                connectionTimeout: 5000,
                socketTimeout: 5000
            }
        });

        // Use just the stream name without the ARN
        this.streams = {
            state: 'IoTMI-notification-stream',
            event: 'IoTmi-device-event-stream',
            lifecycle: 'IoTMI-device-lifecycle-stream'
        };

        this.subscribers = new Map();
        console.log('NotificationService constructed with streams:', this.streams);
    }

    public static getInstance(): NotificationService {
        if (!NotificationService.instance) {
            NotificationService.instance = new NotificationService();
        }
        return NotificationService.instance;
    }

    public async initialize(): Promise<void> {
        if (this.isInitialized) {
            console.log('NotificationService already initialized, skipping...');
            return;
        }

        if (this.initializationPromise) {
            console.log('NotificationService initialization already in progress, waiting...');
            return this.initializationPromise;
        }

        this.initializationPromise = (async () => {
            try {
                console.log('Initializing NotificationService...');
                await this.startEventStreamPolling();
                this.isInitialized = true;
                console.log('NotificationService initialized successfully');
            } catch (error) {
                console.error('Failed to initialize NotificationService:', error);
                this.isInitialized = false;
                throw error;
            } finally {
                this.initializationPromise = null;
            }
        })();

        return this.initializationPromise;
    }

    private getJitteredInterval(): number {
        const jitter = Math.random() * this.MAX_JITTER;
        return Math.min(this.currentBackoff + jitter, this.maxBackoffInterval);
    }

    private handleThroughputExceeded() {
        this.currentBackoff = Math.min(this.currentBackoff * 2, this.maxBackoffInterval);
        console.log(`Throughput exceeded, increasing backoff to ${this.currentBackoff}ms`);
    }

    private resetBackoff() {
        this.currentBackoff = this.BASE_POLLING_INTERVAL;
    }

    private async startEventStreamPolling() {
        // If polling setup is already in progress, wait for it
        if (this.pollingSetupPromise) {
            console.log('Polling setup already in progress, waiting for completion...');
            return this.pollingSetupPromise;
        }

        // If polling is already active, don't start again
        if (this.isPolling) {
            console.log('Polling already active, skipping initialization');
            return;
        }

        // Create a new promise to track this setup attempt
        this.pollingSetupPromise = (async () => {
            try {
                this.isPolling = true;
                console.log('Starting event stream polling');
                
                if (!this.streams.event) {
                    throw new Error('Event stream name is not configured');
                }

                // Log configuration for debugging
                console.log('Kinesis configuration:', {
                    region: process.env.REACT_APP_AWS_REGION,
                    streamName: this.streams.event,
                    shardId: 'shardId-000000000002',
                    credentials: {
                        hasAccessKeyId: !!process.env.REACT_APP_AWS_ACCESS_KEY_ID,
                        hasSecretKey: !!process.env.REACT_APP_AWS_SECRET_ACCESS_KEY,
                        hasSessionToken: !!process.env.REACT_APP_AWS_SESSION_TOKEN
                    }
                });

                try {
                    let isPollingInProgress = false;
                    let currentShardIterator: string | undefined;
                    const MIN_POLL_INTERVAL = 10000; // 10 seconds between polls
                    let lastPollTime = 0;
                    let lastSuccessfulPollTime = Date.now();
                    let lastDataSize = 0; // Track the size of data received in last poll
                    let lastDataTime = Date.now(); // Track when we received the last data

                    // Track when we last did a full state refresh
                    let lastStateRefreshTime = 0;
                    const STATE_REFRESH_INTERVAL = 60000; // Refresh device states every 60 seconds

                    // Start the polling loop in the background
                    const startPolling = () => {
                        if (!this.isPolling) {
                            console.log('Polling stopped, not restarting');
                            return;
                        }

                        // Track when we last did a full state refresh
                        let lastStateRefreshTime = 0;
                        const STATE_REFRESH_INTERVAL = 60000; // Refresh device states every 60 seconds

                        const poll = async () => {
                            if (!this.isPolling || isPollingInProgress) {
                                console.log('Skipping poll - polling stopped or already in progress');
                                return;
                            }

                            try {
                                isPollingInProgress = true;

                                // DISABLED: Periodic device state refresh to reduce cloud load
                                // Let page-based polling handle state updates instead
                                console.log('Periodic device state refresh disabled - using page-based polling');

                                if (!currentShardIterator) {
                                    console.log('No valid iterator, getting new one...');
        try {
            const iteratorCommand = new GetShardIteratorCommand({
                                            StreamName: this.streams.event,
                                            ShardId: 'shardId-000000000002',
                ShardIteratorType: 'LATEST'
                                        });
                                        const iteratorResponse = await this.kinesisClient.send(iteratorCommand);
                                        currentShardIterator = iteratorResponse.ShardIterator;
                                        console.log('Successfully got new shard iterator');
                                    } catch (error) {
                                        console.error('Failed to get new shard iterator:', error);
                                        setTimeout(poll, MIN_POLL_INTERVAL);
                                        return;
                                    }
                                }

                                // Check if we need to wait due to data read rate limits
                                const timeSinceLastData = Date.now() - lastDataTime;
                                const timeSinceLastPoll = Date.now() - lastPollTime;

                                if (lastDataSize > 1000 && timeSinceLastData < 10000) { // 10s after big responses
                                    console.log('Rate limiting after large response, waiting...');
                                    setTimeout(poll, 10000 - timeSinceLastData);
                                    return;
                                }

                                if (timeSinceLastPoll < MIN_POLL_INTERVAL) {
                                    console.log('Rate limiting standard polling, waiting...');
                                    setTimeout(poll, MIN_POLL_INTERVAL - timeSinceLastPoll);
                                    return;
                                }

                                const recordsCommand = new GetRecordsCommand({
                                    ShardIterator: currentShardIterator,
                                    Limit: 100
                                });

                                lastPollTime = Date.now();
                                let response;
                                try {
                                    response = await this.kinesisClient.send(recordsCommand);
                                    lastSuccessfulPollTime = Date.now();
                                    console.log('GetRecords call completed successfully');
                                    
                                    if (!response?.NextShardIterator) {
                                        console.log('No NextShardIterator returned, but continuing with current iterator as there might be no new records');
                                        // Keep using the current iterator instead of getting a new one
                                        // This is normal when there are no new records
                                        console.log('Scheduling next poll with current iterator');
                                        setTimeout(poll, MIN_POLL_INTERVAL);
                                        return;
                                    }

                                    // Update the current shard iterator with the next one
                                    currentShardIterator = response.NextShardIterator;
                                    console.log('Updated shard iterator for next poll');

                                    // Check for records
                                    if (response.Records && response.Records.length > 0) {
                                        console.log(`Received ${response.Records.length} records`);
                                        
                                        // Update tracking
                                        lastDataTime = Date.now();
                                        // Estimate size based on record count (rough approximation)
                                        lastDataSize = response.Records.length;
                                        
                                        for (const record of response.Records) {
                                            if (!record.Data || !record.SequenceNumber) {
                                                console.warn('Skipping record with missing data or sequence number');
                                                continue;
                                            }

                                            try {
                                                // Generate a unique ID for this record to avoid processing duplicates
                                                const recordId = record.SequenceNumber;

                                                // Check if we've already processed this record
                                                if (this.processedRecords.has(recordId)) {
                                                    console.log(`Skipping already processed record: ${recordId}`);
                                                    continue;
                                                }

                                                // Decode and parse the record data
                                                const recordData = new TextDecoder().decode(record.Data);
                                                const message = JSON.parse(recordData);
                                                console.log('Decoded message:', message);

                                                // Check for required fields
                                                if (!message.messageType || !message.resources || message.resources.length === 0) {
                                                    console.warn('Skipping invalid message (missing type or resources)');
                                                    continue;
                                                }

                                                // Extract device ID from resources (assuming format: 'device:deviceId')
                                                const deviceIdResource = message.resources.find((r: string) => r.startsWith('device:'));
                                                if (!deviceIdResource) {
                                                    console.warn('Skipping message with no device resource');
                                                    continue;
                                                }

                                                const deviceId = deviceIdResource.split(':')[1];
                                                if (!deviceId) {
                                                    console.warn('Skipping message with invalid device ID');
                                                    continue;
                                                }

                                                // Process message based on type
                                                console.log(`Processing message of type ${message.messageType} for device ${deviceId}`);

                                                // Add to processed records
                                                this.processedRecords.add(recordId);

                                                // Maintain sliding window of processed records
                                                if (this.processedRecords.size > this.MAX_PROCESSED_RECORDS) {
                                                    const oldestRecord = Array.from(this.processedRecords)[0];
                                                    this.processedRecords.delete(oldestRecord);
                                                }

                                                if (message.messageType === 'DEVICE_EVENT') {
                                                    await this.handleEventMessage(deviceId, message);
                                                }

                                            } catch (error) {
                                                console.error('Error processing record:', error);
                                                // Continue with next record instead of breaking the loop
                                                continue;
                                            }
                                        }
                                    } else {
                                        console.log('No new records in this poll');
                                        // Reset data size tracking when no records
                                        lastDataSize = 0;
                                    }

                                    // Schedule next poll
                                    setTimeout(poll, MIN_POLL_INTERVAL);

                                } catch (error) {
                                    console.error('Error in GetRecords call:', error);
                                    const now = Date.now();
                                    
                                    // If it's been too long since we got a valid response, get a new iterator
                                    if (now - lastSuccessfulPollTime > 60000) { // 1 minute
                                        console.log('Too long since last successful poll, clearing iterator');
                                        // Don't clear the iterator, just retry with current one
                                        console.log('Retrying with current iterator');
                                    }
                                    
                                    // Schedule next poll with delay
                                    setTimeout(poll, MIN_POLL_INTERVAL);
                                } finally {
                                    isPollingInProgress = false;
                                }
                            } catch (error) {
                                console.error('Error in polling loop:', error);
                                setTimeout(poll, MIN_POLL_INTERVAL);
                                isPollingInProgress = false;
                            }
                        };

                        // DISABLED: One-time read of device states to reduce cloud load
                        // Let page-based polling handle initial state loading instead
                        console.log('Initial readAllDeviceStates disabled - using page-based polling');
                        
                        // Start polling loop
                        poll();
                    };

                    // Start the polling
                    startPolling();
                } catch (error) {
                    console.error('Error setting up polling loop:', error);
                    this.isPolling = false;
                    throw error;
                }
            } catch (error) {
                console.error('Error setting up event stream polling:', error);
                this.isPolling = false;
                // Instead of retrying automatically, we'll let the caller decide
                throw error;
            } finally {
                this.pollingSetupPromise = null;
            }
        })();

        return this.pollingSetupPromise;
    }

    private extractDeviceId(resourceArn: string): string | null {
        if (!resourceArn) return null;
        const parts = resourceArn.split('/');
        return parts[parts.length - 1] || null;
    }

    private async handleEventMessage(deviceId: string, message: DeviceEventMessage) {
        console.log(`Processing event for device ${deviceId}:`, message);

        if (message.messageType === 'DEVICE_EVENT') {
            const newState: DeviceState = {
                Endpoints: message.payload.endpoints.map(endpoint => ({
                    endpointId: endpoint.endpointId,
                    deviceTypes: [],
                    capabilities: endpoint.capabilities.map(capability => ({
                        id: capability.id,
                        name: capability.name,
                        version: capability.version,
                        properties: capability.properties.map(prop => ({
                            name: prop.name,
                            value: {
                                propertyValue: prop.value,
                                lastChangedAt: this.convertTimestamp(message.timestamp)
                            }
                        })),
                        actions: [],
                        events: []
                    }))
                }))
            };

            // Get current state from Redux store
            const currentState = store.getState().devices.devices.find(
                d => d.ManagedThingId === deviceId
            )?.currentState || null;

            console.log('Current state from Redux:', {
                deviceId,
                currentState,
                timestamp: new Date().toISOString()
            });
            console.log('New state from Kinesis:', {
                deviceId,
                newState,
                timestamp: new Date().toISOString()
            });

            // Compare and merge states based on timestamps
            const mergedState = this.mergeStates(currentState, newState);
            console.log('Merged state:', {
                deviceId,
                mergedState,
                timestamp: new Date().toISOString()
            });

            // Update device state in Redux store
            store.dispatch(updateDeviceState({ 
                ManagedThingId: deviceId, 
                newState: mergedState 
            }));

            // Notify subscribers
            const deviceSubscribers = this.subscribers.get(deviceId);
            if (deviceSubscribers) {
                console.log(`Notifying ${deviceSubscribers.size} subscribers for device ${deviceId}`);
                deviceSubscribers.forEach(callback => {
                    try {
                        callback(mergedState);
                        console.log(`Successfully notified subscriber for device ${deviceId}`);
                    } catch (error) {
                        console.error(`Error notifying subscriber for device ${deviceId}:`, error);
                    }
                });
            }

            console.log('Updated Redux store and notified subscribers for device:', {
                deviceId,
                timestamp: new Date().toISOString()
            });
            
            // DISABLED: Proactive read state after events to reduce cloud load
            // Let page-based polling handle state updates instead
            console.log(`Event processed for device ${deviceId} - proactive read disabled`);
        }
    }

    private async proactiveReadState(deviceId: string) {
        try {
            console.log(`Proactively reading state for device ${deviceId}`);
            
            // Get current device state to extract endpoints and capabilities
            const deviceState = await deviceService.getDeviceState(deviceId);
            
            if (!deviceState || !deviceState.Endpoints || deviceState.Endpoints.length === 0) {
                console.log(`No endpoints found for device ${deviceId}, skipping proactive read`);
                return;
            }
            
            // Properly format endpoints for the SendManagedThingCommandCommand
            const formattedEndpoints: CommandEndpoint[] = deviceState.Endpoints
                .filter(endpoint => endpoint.capabilities && endpoint.capabilities.length > 0)
                .map(endpoint => ({
                    endpointId: endpoint.endpointId,
                    capabilities: endpoint.capabilities.map(capability => ({
                        id: capability.id,
                        name: capability.name || capability.id.split('@')[0].split('.')[1],
                        version: capability.version || "1",
                        actions: [
                            {
                                name: "ReadState",
                                parameters: {
                                    propertiesToRead: ["*"]
                                }
                            }
                        ]
                    }))
                }));
            
            if (formattedEndpoints.length === 0) {
                console.log(`No valid endpoints to read for device ${deviceId}`);
                return;
            }
            
            console.log(`Sending ReadState commands for device ${deviceId} with formatted payload:`, 
                JSON.stringify(formattedEndpoints, null, 2));
                
            try {
                // Use direct command approach for proper formatting
                await deviceService.client.send(new SendManagedThingCommandCommand({
                    ManagedThingId: deviceId,
                    Endpoints: formattedEndpoints
                }));
                console.log(`Successfully sent ReadState command for device ${deviceId}`);
            } catch (error) {
                console.error(`Error sending ReadState command for device ${deviceId}:`, error);
                
                // Fallback to individual executeAction calls if the batch command fails
                console.log(`Falling back to individual capability reads for device ${deviceId}`);
                for (const endpoint of deviceState.Endpoints) {
                    if (!endpoint.capabilities || endpoint.capabilities.length === 0) continue;
                    
                    for (const capability of endpoint.capabilities) {
                        try {
                            await deviceService.executeAction(
                                deviceId,
                                capability.id,
                                'ReadState',
                                endpoint.endpointId,
                                { propertiesToRead: ['*'] }
                            );
                            await new Promise(resolve => setTimeout(resolve, 100)); // Small delay
                        } catch (innerError) {
                            console.error(`Fallback read failed for capability ${capability.id}:`, innerError);
                        }
                    }
                }
            }
            
            console.log(`Completed proactive read state for device ${deviceId}`);
        } catch (error) {
            console.error(`Error in proactive read state for device ${deviceId}:`, error);
        }
    }

    private mergeStates(currentState: DeviceState | null, newState: DeviceState): DeviceState {
        if (!currentState) {
            console.log('No current state, using new state as is');
            return newState;
        }

        const mergedState = {
            Endpoints: newState.Endpoints.map(newEndpoint => {
                const currentEndpoint = currentState.Endpoints.find(
                    e => e.endpointId === newEndpoint.endpointId
                );

                if (!currentEndpoint) {
                    console.log(`No current endpoint found for ${newEndpoint.endpointId}, using new endpoint`);
                    return newEndpoint;
                }

                return {
                    ...newEndpoint,
                    capabilities: newEndpoint.capabilities.map(newCapability => {
                        const currentCapability = currentEndpoint.capabilities.find(
                            c => c.id === newCapability.id
                        );

                        if (!currentCapability) {
                            console.log(`No current capability found for ${newCapability.id}, using new capability`);
                            return newCapability;
                        }

                        return {
                            ...newCapability,
                            properties: newCapability.properties.map(newProp => {
                                const currentProp = currentCapability.properties.find(
                                    p => p.name === newProp.name
                                );

                                if (!currentProp) {
                                    console.log(`No current property found for ${newProp.name}, using new property`);
                                    return newProp;
                                }

                                // Ensure both values and lastChangedAt exist
                                if (!newProp.value?.lastChangedAt || !currentProp.value?.lastChangedAt) {
                                    console.log(`Missing lastChangedAt for property ${newProp.name}, using new property`);
                                    return newProp;
                                }

                                const newTimestamp = new Date(newProp.value.lastChangedAt).getTime();
                                const currentTimestamp = new Date(currentProp.value.lastChangedAt).getTime();

                                // Only use the newer value
                                const result = newTimestamp > currentTimestamp ? newProp : currentProp;
                                console.log(`Property ${newProp.name} merge result:`, {
                                    newTimestamp: new Date(newTimestamp).toISOString(),
                                    currentTimestamp: new Date(currentTimestamp).toISOString(),
                                    usingNewer: newTimestamp > currentTimestamp
                                });
                                return result;
                            })
                        };
                    })
                };
            })
        };

        console.log('State merge completed:', {
            deviceId: newState.Endpoints[0]?.endpointId,
            endpointCount: mergedState.Endpoints.length,
            timestamp: new Date().toISOString()
        });

        return mergedState;
    }

    public subscribeToDeviceUpdates(deviceId: string, callback: (state: DeviceState) => void): void {
        if (!this.subscribers.has(deviceId)) {
            this.subscribers.set(deviceId, new Set());
        }
        this.subscribers.get(deviceId)?.add(callback);
        console.log(`Subscribed to updates for device ${deviceId}`);
    }

    public unsubscribeFromDeviceUpdates(deviceId: string, callback: (state: DeviceState) => void): void {
        const deviceSubscribers = this.subscribers.get(deviceId);
        if (deviceSubscribers) {
            deviceSubscribers.delete(callback);
            if (deviceSubscribers.size === 0) {
                this.subscribers.delete(deviceId);
            }
            console.log(`Unsubscribed from updates for device ${deviceId}`);
        }
    }

    public disconnect(): void {
        if (!this.isPolling) {
            console.log('NotificationService already disconnected');
            return;
        }

        console.log('Disconnecting NotificationService...');
        this.isPolling = false;
        if (this.pollingTimeout) {
            clearTimeout(this.pollingTimeout);
            this.pollingTimeout = null;
        }
        this.subscribers.clear();
        this.processedRecords.clear();
        this.isInitialized = false;
        console.log('NotificationService disconnected');
    }

    private convertTimestamp(timestamp: string | number): string {
        // If timestamp is a string that's all digits, convert to number
        if (typeof timestamp === 'string' && /^\d+$/.test(timestamp)) {
            timestamp = parseInt(timestamp, 10);
        }

        // If timestamp is a number and appears to be in milliseconds (13 digits), use as is
        // If it's in seconds (10 digits), multiply by 1000
        if (typeof timestamp === 'number') {
            const timestampStr = timestamp.toString();
            if (timestampStr.length === 10) {
                timestamp *= 1000;
            }
            return new Date(timestamp).toISOString();
        }

        // If it's already a valid ISO string, return as is
        try {
            return new Date(timestamp).toISOString();
        } catch (error) {
            console.error('Invalid timestamp:', timestamp);
            return new Date().toISOString(); // Fallback to current time
        }
    }

    private async readAllDeviceStates() {
        try {
            // Get all devices from store
            const devices = store.getState().devices.devices;
            console.log(`Reading states for ${devices.length} devices`);
            
            for (const device of devices) {
                await this.proactiveReadState(device.ManagedThingId);
                // Add delay between devices
                await new Promise(resolve => setTimeout(resolve, 200));
            }
        } catch (error) {
            console.error('Error reading all device states:', error);
        }
    }
}

// Export singleton instance
export const notificationService = NotificationService.getInstance();