import { DeviceState } from '../types/deviceTypes';
import { deviceService } from './deviceService';
import { store } from '../store/store';
import { updateDeviceState } from '../store/deviceSlice';
import { debugLogger } from '../utils/debugLogger';
import { 
    SendManagedThingCommandCommand,
    CommandEndpoint,
    CommandCapability
} from '@aws-sdk/client-iot-managed-integrations';

export class DeviceStatePollingService {
    private static instance: DeviceStatePollingService | null = null;
    private readStateIntervals: Map<string, NodeJS.Timeout> = new Map(); // 1-second ReadState intervals
    private getStateIntervals: Map<string, NodeJS.Timeout> = new Map(); // 5-second GetState intervals
    private subscribers: Map<string, Set<(state: DeviceState) => void>> = new Map();
    private isInitialized: boolean = false;
    private initializationPromise: Promise<void> | null = null;
    private readonly POLLING_INTERVAL = 5000; // 5 seconds for dashboard
    private readonly DEVICE_DETAIL_POLLING_INTERVAL = 5000; // 5 seconds for device details
    private readonly MAX_CONCURRENT_REQUESTS = 20; // Increased from 10 to allow more parallel processing
    private requestQueue: Array<() => Promise<void>> = [];
    private activeRequests: number = 0;
    private lastReadStateTime: Map<string, number> = new Map(); // Track last ReadState time per device
    private deviceCapabilitiesCache: Map<string, { capabilities: any; timestamp: number }> = new Map(); // Cache capabilities
    private readonly CAPABILITIES_CACHE_DURATION = 300000; // 5 minutes cache for capabilities
    
    // Track last command timestamps per device to prevent stale state overwrites
    private lastCommandTimestamps: Map<string, number> = new Map();
    
    // Track consecutive command counts for adaptive thresholds
    private consecutiveCommandCounts: Map<string, number> = new Map();
    private lastCommandWindowStart: Map<string, number> = new Map();
    
    // Track expected device states based on toggle sequence
    private pendingStateChanges: Map<string, number> = new Map(); // deviceId -> number of pending state changes
    private lastKnownState: Map<string, boolean> = new Map(); // deviceId -> last known device state
    
    // Page-based polling control
    private currentPollingMode: 'dashboard' | 'device-detail' | 'onboarding' | 'stopped' = 'stopped';
    private focusedDeviceId: string | null = null;

    // Define interested properties for each capability type
    private readonly CAPABILITY_INTERESTED_PROPERTIES: Record<string, string[]> = {
        'matter.OnOff@1.4': ['OnOff'],
        'matter.LevelControl@1.4': ['CurrentLevel', 'MinLevel', 'MaxLevel'],
        'matter.ColorControl@1.4': ['CurrentHue', 'CurrentSaturation', 'CurrentX', 'CurrentY', 'ColorTemperatureMireds']
    };

    private constructor() {
        debugLogger.log('DeviceStatePollingService constructed');
    }

    public static getInstance(): DeviceStatePollingService {
        if (!DeviceStatePollingService.instance) {
            DeviceStatePollingService.instance = new DeviceStatePollingService();
        }
        return DeviceStatePollingService.instance;
    }

    public async initialize(): Promise<void> {
        if (this.isInitialized) {
            console.log('DeviceStatePollingService already initialized - POLLING DISABLED');
            return;
        }

        if (this.initializationPromise) {
            return this.initializationPromise;
        }

        this.initializationPromise = (async () => {
            try {
                console.log('DeviceStatePollingService initialized - POLLING DISABLED FOR CLOUD LOAD REDUCTION');
                // DISABLED: No automatic polling to reduce cloud load
                // Manual polling can be enabled per page if needed
                this.isInitialized = true;
                console.log('DeviceStatePollingService ready (no automatic polling)');
            } catch (error) {
                console.error('Failed to initialize DeviceStatePollingService:', error);
                this.isInitialized = false;
                throw error;
            } finally {
                this.initializationPromise = null;
            }
        })();

        return this.initializationPromise;
    }

    private async processQueue(): Promise<void> {
        if (this.activeRequests >= this.MAX_CONCURRENT_REQUESTS) {
            console.log(`Queue processing paused - at max concurrent limit (${this.activeRequests}/${this.MAX_CONCURRENT_REQUESTS}). Queue length: ${this.requestQueue.length}`);
            return;
        }

        const request = this.requestQueue.shift();
        if (!request) {
            return;
        }

        this.activeRequests++;
        console.log(`Starting request - Concurrent: ${this.activeRequests}/${this.MAX_CONCURRENT_REQUESTS}, Queue length: ${this.requestQueue.length}`);
        const startTime = Date.now();
        
        try {
            await request();
        } finally {
            this.activeRequests--;
            const duration = Date.now() - startTime;
            console.log(`Request completed in ${duration}ms. Active requests: ${this.activeRequests}/${this.MAX_CONCURRENT_REQUESTS}, Queue length: ${this.requestQueue.length}`);
        }

        // Process next request immediately if available
        this.processQueue();
    }

    private async pollDeviceState(deviceId: string): Promise<void> {
        const startTime = Date.now();
        let device: any = null;
        
        try {
            console.log(`Starting simplified polling cycle for device ${deviceId}`);
            
            // Step 1: Get device info for logging
            device = await deviceService.getDevice(deviceId);
            console.log(`Device info retrieved for ${deviceId} (${device?.Protocol})`);
            
            // Step 2: Get the current state directly from cloud (no ReadState needed)
            console.log(`Getting current state for device ${deviceId}`);
            const state = await deviceService.getDeviceState(deviceId);
            
            const duration = Date.now() - startTime;
            console.log(`Completed simplified polling cycle for device ${deviceId} in ${duration}ms (${device?.Protocol})`);

            if (state) {
                // Log before updating Redux store
                const onOffCapability = state.Endpoints?.[0]?.capabilities?.find(c => c.name === 'On/Off' || c.name === 'OnOff');
                const onOffValue = onOffCapability?.properties?.find(p => p.name === 'OnOff')?.value?.propertyValue;
                console.log(`REDUX UPDATE - Device: ${deviceId}, OnOff: ${onOffValue}, Protocol: ${device?.Protocol}, Timestamp: ${new Date().toISOString()}`);
                
                // Update Redux store
                store.dispatch(updateDeviceState({ ManagedThingId: deviceId, newState: state }));

                // Notify subscribers - state is guaranteed to be non-null here
                const deviceSubscribers = this.subscribers.get(deviceId);
                if (deviceSubscribers) {
                    deviceSubscribers.forEach(callback => {
                        try {
                            callback(state!); // Use non-null assertion since we're inside the if (state) block
                        } catch (error) {
                            console.error(`Error notifying subscriber for device ${deviceId}:`, error);
                        }
                    });
                }
            } else {
                console.warn(`No state returned for device ${deviceId}`);
            }
        } catch (error) {
            console.error(`Error in simplified polling cycle for device ${deviceId}:`, error);
        }
    }


    /**
     * Get device capabilities with caching (5 minutes cache)
     */
    private async getCachedDeviceCapabilities(deviceId: string) {
        const now = Date.now();
        const cached = this.deviceCapabilitiesCache.get(deviceId);
        
        // Return cached capabilities if still valid (within 5 minutes)
        if (cached && (now - cached.timestamp) < this.CAPABILITIES_CACHE_DURATION) {
            console.log(`Using cached capabilities for device ${deviceId} (age: ${Math.round((now - cached.timestamp) / 1000)}s)`);
            return cached.capabilities;
        }
        
        // Fetch fresh capabilities and cache them
        console.log(`Fetching fresh capabilities for device ${deviceId} (cache ${cached ? 'expired' : 'missing'})`);
        const capabilities = await deviceService.getDeviceCapabilities(deviceId);
        this.deviceCapabilitiesCache.set(deviceId, { capabilities, timestamp: now });
        
        return capabilities;
    }

    public async startPolling(deviceId: string): Promise<void> {
        if (this.getStateIntervals.has(deviceId)) {
            console.log(`Already polling device ${deviceId}`);
            return;
        }

        console.log(`Starting simplified polling for device ${deviceId} (GetState only: 5s)`);
        
        // Immediately get initial state
        this.requestQueue.push(async () => {
            try {
                const initialState = await deviceService.getDeviceState(deviceId);
                if (initialState) {
                    store.dispatch(updateDeviceState({ 
                        ManagedThingId: deviceId, 
                        newState: initialState 
                    }));
                    
                    const deviceSubscribers = this.subscribers.get(deviceId);
                    if (deviceSubscribers) {
                        deviceSubscribers.forEach(callback => {
                            try {
                                callback(initialState);
                            } catch (error) {
                                console.error(`Error notifying subscriber for device ${deviceId}:`, error);
                            }
                        });
                    }
                }
            } catch (error) {
                console.error(`Error in initial poll for device ${deviceId}:`, error);
            }
        });
        this.processQueue();
        
        // Set up simplified GetState interval (every 5 seconds for device details)
        const getStateInterval = setInterval(() => {
            this.requestQueue.push(async () => {
                try {
                    console.log(`Starting simplified GetState cycle for device ${deviceId}`);
                    
                    // Get current state directly from cloud
                    console.log(`Getting current state for device ${deviceId}`);
                    const state = await deviceService.getDeviceState(deviceId);
                    
                    if (state) {
                        // Log GetState response timing for comparison with command timing
                        const onOffCapability = state.Endpoints?.[0]?.capabilities?.find(c => c.name === 'On/Off' || c.name === 'OnOff');
                        const onOffValue = onOffCapability?.properties?.find(p => p.name === 'OnOff')?.value?.propertyValue;
                        const lastChangedAt = onOffCapability?.properties?.find(p => p.name === 'OnOff')?.value?.lastChangedAt;
                        const lastChangedTimestamp = lastChangedAt ? new Date(lastChangedAt).getTime() : 0;
                        
                        console.log(`GETSTATE_RECEIVED: Device ${deviceId}, OnOff ${onOffValue}, LastChangedAt ${lastChangedAt}, LastChangedTimestamp ${lastChangedTimestamp}, ReceivedAt ${Date.now()}`);
                        
                        if (this.shouldUpdateState(deviceId, state)) {
                            console.log(`REDUX UPDATE - Device: ${deviceId}, OnOff: ${onOffValue}, Timestamp: ${new Date().toISOString()}`);
                            
                            store.dispatch(updateDeviceState({ ManagedThingId: deviceId, newState: state }));

                            const deviceSubscribers = this.subscribers.get(deviceId);
                            if (deviceSubscribers) {
                                deviceSubscribers.forEach(callback => {
                                    try {
                                        callback(state);
                                    } catch (error) {
                                        console.error(`Error notifying subscriber for device ${deviceId}:`, error);
                                    }
                                });
                            }
                        } else {
                            console.log(`Skipped Redux update for device ${deviceId} due to stale state`);
                        }
                    }
                    console.log(`Completed simplified GetState cycle for device ${deviceId}`);
                } catch (error) {
                    console.error(`Error in simplified GetState cycle for device ${deviceId}:`, error);
                }
            });
            this.processQueue();
        }, this.DEVICE_DETAIL_POLLING_INTERVAL);

        this.getStateIntervals.set(deviceId, getStateInterval);
    }

    public stopPolling(deviceId: string): void {
        const readStateInterval = this.readStateIntervals.get(deviceId);
        const getStateInterval = this.getStateIntervals.get(deviceId);
        
        if (readStateInterval) {
            clearInterval(readStateInterval);
            this.readStateIntervals.delete(deviceId);
            console.log(`Stopped ReadState polling for device ${deviceId}`);
        }
        
        if (getStateInterval) {
            clearInterval(getStateInterval);
            this.getStateIntervals.delete(deviceId);
            console.log(`Stopped GetState polling for device ${deviceId}`);
        }
    }

    public subscribeToDeviceUpdates(deviceId: string, callback: (state: DeviceState) => void): void {
        if (!this.subscribers.has(deviceId)) {
            this.subscribers.set(deviceId, new Set());
        }
        this.subscribers.get(deviceId)?.add(callback);
        console.log(`Subscribed to updates for device ${deviceId}`);
    }

    public unsubscribeFromDeviceUpdates(deviceId: string, callback: (state: DeviceState) => void): void {
        const deviceSubscribers = this.subscribers.get(deviceId);
        if (deviceSubscribers) {
            deviceSubscribers.delete(callback);
            if (deviceSubscribers.size === 0) {
                this.subscribers.delete(deviceId);
            }
            console.log(`Unsubscribed from updates for device ${deviceId}`);
        }
    }

    /**
     * Record when a command was sent to a device to prevent stale state overwrites
     * Also tracks consecutive commands for adaptive thresholds
     */
    public recordCommand(deviceId: string, timestamp: number = Date.now()): void {
        this.lastCommandTimestamps.set(deviceId, timestamp);
        
        // Track consecutive commands within a 10-second window
        const windowStart = this.lastCommandWindowStart.get(deviceId) || timestamp;
        const windowDuration = timestamp - windowStart;
        
        if (windowDuration > 10000) {
            // New window - reset counter
            this.consecutiveCommandCounts.set(deviceId, 1);
            this.lastCommandWindowStart.set(deviceId, timestamp);
        } else {
            // Within window - increment counter
            const currentCount = this.consecutiveCommandCounts.get(deviceId) || 0;
            this.consecutiveCommandCounts.set(deviceId, currentCount + 1);
        }
        
        const consecutiveCount = this.consecutiveCommandCounts.get(deviceId) || 1;
        console.log(`Recorded command timestamp for device ${deviceId}: ${new Date(timestamp).toISOString()}, consecutive commands: ${consecutiveCount}`);
    }

    /**
     * Record a state change command and track pending state changes
     */
    public recordToggle(deviceId: string, fromState: boolean, toState: boolean, timestamp: number = Date.now()): void {
        // Record the command timestamp
        this.recordCommand(deviceId, timestamp);
        
        // Only count actual state changes, not redundant commands
        if (fromState === toState) {
            console.log(`No state change for device ${deviceId}: ${fromState} → ${toState} (redundant command ignored)`);
            return;
        }
        
        // Set the baseline state if we don't have one
        if (!this.lastKnownState.has(deviceId)) {
            this.lastKnownState.set(deviceId, fromState);
        }
        
        // Increment pending state changes
        const currentPending = this.pendingStateChanges.get(deviceId) || 0;
        this.pendingStateChanges.set(deviceId, currentPending + 1);
        
        console.log(`State change recorded for device ${deviceId}:`, {
            fromState,
            toState,
            pendingStateChanges: currentPending + 1,
            timestamp: new Date(timestamp).toISOString()
        });
    }

    /**
     * Process device state change and count down pending state changes
     */
    private processToggleResponse(deviceId: string, deviceState: boolean): boolean {
        const pendingChanges = this.pendingStateChanges.get(deviceId) || 0;
        const lastKnownState = this.lastKnownState.get(deviceId);
        
        if (pendingChanges === 0) {
            // No pending state changes, accept any state
            this.lastKnownState.set(deviceId, deviceState);
            return true;
        }
        
        // Check if this is a state change from what we last knew
        if (lastKnownState !== undefined && lastKnownState !== deviceState) {
            // This is a state change! Decrement pending changes
            const newPendingChanges = pendingChanges - 1;
            this.pendingStateChanges.set(deviceId, newPendingChanges);
            this.lastKnownState.set(deviceId, deviceState);
            
            if (newPendingChanges === 0) {
                // All state changes processed - accept this final state
                console.log(`State change sequence complete for device ${deviceId}:`, {
                    deviceState,
                    lastKnownState,
                    completedChanges: pendingChanges,
                    result: 'accepting-final-state'
                });
                return true;
            } else {
                // More state changes expected - accept this intermediate state but keep waiting
                console.log(`State change ${pendingChanges - newPendingChanges}/${pendingChanges} for device ${deviceId}:`, {
                    deviceState,
                    lastKnownState,
                    remainingChanges: newPendingChanges,
                    result: 'accepting-intermediate-state'
                });
                return true;
            }
        } else {
            // No state change detected - ignore this response
            console.log(`No state change detected for device ${deviceId}:`, {
                deviceState,
                lastKnownState,
                pendingChanges,
                result: 'waiting-for-state-change'
            });
            return false;
        }
    }

    /**
     * Get the last command timestamp for a device
     */
    public getLastCommandTimestamp(deviceId: string): number | undefined {
        return this.lastCommandTimestamps.get(deviceId);
    }

    /**
     * Check if state data should update - SIMPLIFIED for Zigbee lights
     */
    private shouldUpdateState(deviceId: string, stateData: any): boolean {
        const lastCommandTime = this.lastCommandTimestamps.get(deviceId);
        if (!lastCommandTime) {
            // No command recorded, always allow update
            return true;
        }

        const timeSinceCommand = Date.now() - lastCommandTime;
        
        // For Zigbee devices that return empty endpoints, be more permissive
        const hasEmptyEndpoints = !stateData?.Endpoints || stateData.Endpoints.length === 0;
        if (hasEmptyEndpoints) {
            // If AWS returns empty endpoints (common with Zigbee), use shorter grace period
            if (timeSinceCommand > 3000) { // 3 seconds for fallback data
                console.log(`Accepting fallback state for Zigbee device ${deviceId} (empty endpoints):`, {
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    reason: 'zigbee-fallback-grace-period'
                });
                return true;
            } else {
                console.log(`Waiting for Zigbee device ${deviceId} (empty endpoints):`, {
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    reason: 'within-zigbee-grace-period'
                });
                return false;
            }
        }

        // For devices with actual state data, use normal logic
        const gracePeriodExpired = timeSinceCommand > 8000; // 8-second grace period
        
        if (gracePeriodExpired) {
            console.log(`Accepting state due to grace period expiry for device ${deviceId}:`, {
                timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                reason: 'grace-period-expired'
            });
            return true;
        }

        // Find the most recent lastChangedAt timestamp in the state data
        let mostRecentStateTime = 0;
        if (stateData?.Endpoints) {
            for (const endpoint of stateData.Endpoints) {
                if (endpoint.capabilities) {
                    for (const capability of endpoint.capabilities) {
                        if (capability.properties) {
                            for (const property of capability.properties) {
                                if (property.value?.lastChangedAt) {
                                    const stateTime = new Date(property.value.lastChangedAt).getTime();
                                    mostRecentStateTime = Math.max(mostRecentStateTime, stateTime);
                                }
                            }
                        }
                    }
                }
            }
        }

        const deviceStateIsNewer = mostRecentStateTime > lastCommandTime;
        const timeSinceCommandSeconds = Math.round(timeSinceCommand / 1000);
        
        if (deviceStateIsNewer && timeSinceCommand > 2000) {
            console.log(`Accepting newer state for device ${deviceId}:`, {
                lastCommandTime: new Date(lastCommandTime).toISOString(),
                mostRecentStateTime: new Date(mostRecentStateTime).toISOString(),
                timeSinceCommand: `${timeSinceCommandSeconds}s`,
                reason: 'newer-state-and-threshold-met'
            });
            return true;
        } else {
            console.log(`Waiting for device ${deviceId}:`, {
                lastCommandTime: new Date(lastCommandTime).toISOString(),
                mostRecentStateTime: new Date(mostRecentStateTime).toISOString(),
                timeSinceCommand: `${timeSinceCommandSeconds}s`,
                deviceStateIsNewer,
                reason: deviceStateIsNewer ? 'within-2s-threshold' : 'stale-timestamp'
            });
            return false;
        }
    }

    /**
     * Set polling mode based on current page/screen
     */
    public async setPollingMode(mode: 'dashboard' | 'device-detail' | 'onboarding' | 'stopped', focusedDeviceId?: string): Promise<void> {
        console.log(`POLLING MODE CHANGE: ${this.currentPollingMode} → ${mode}${focusedDeviceId ? ` (focused device: ${focusedDeviceId})` : ''}`);
        const readStateDevices = Array.from(this.readStateIntervals.keys());
        const getStateDevices = Array.from(this.getStateIntervals.keys());
        const activeDevices = [...readStateDevices, ...getStateDevices];
        console.log(`Current active devices BEFORE mode change: [${activeDevices.join(', ')}]`);
        
        const previousMode = this.currentPollingMode;
        const previousFocusedDevice = this.focusedDeviceId;
        
        this.currentPollingMode = mode;
        this.focusedDeviceId = focusedDeviceId || null;
        
        // ALWAYS stop all polling first, regardless of mode
        this.stopAllDevicePolling();
        
        // Wait a moment to ensure cleanup
        await new Promise(resolve => setTimeout(resolve, 100));
        
        console.log(`Starting new polling mode: ${mode}`);
        
        // Handle mode transitions
        switch (mode) {
            case 'dashboard':
                // Poll all devices when on dashboard
                await this.startDashboardPolling();
                break;
                
            case 'device-detail':
                // Only poll the focused device
                if (focusedDeviceId) {
                    await this.startSingleDevicePolling(focusedDeviceId);
                }
                break;
                
            case 'onboarding':
                // Stop all device polling during onboarding (already stopped above)
                console.log('Onboarding mode - no device polling');
                break;
                
            case 'stopped':
                // Stop all polling (already stopped above)
                console.log('Stopped mode - no polling');
                break;
        }
        
        const finalActiveDevices = [...Array.from(this.readStateIntervals.keys()), ...Array.from(this.getStateIntervals.keys())];
        console.log(`POLLING MODE CHANGE COMPLETE: Active devices now: [${finalActiveDevices.join(', ')}]`);
    }

    /**
     * Start polling all devices (dashboard mode)
     * Every 5 minutes: ListManagedThings + get capabilities once per device
     * Every 1 minute: GetState for each device (simplified - no ReadState needed)
     */
    private async startDashboardPolling(): Promise<void> {
        console.log('Starting dashboard polling - ListManagedThings every 5min, GetState every 5s');
        
        // Stop any existing polling first
        this.stopAllDevicePolling();
        
        // Start the dashboard discovery and polling cycle
        await this.performDashboardDiscoveryAndSetup();
        
        // Set up 5-minute interval for device discovery and capability refresh
        const discoveryInterval = setInterval(async () => {
            await this.performDashboardDiscoveryAndSetup();
        }, 300000); // 5 minutes
        
        // Store the discovery interval (we'll use readStateIntervals map for this)
        this.readStateIntervals.set('dashboard-discovery', discoveryInterval);
    }

    /**
     * Perform dashboard device discovery and setup polling
     */
    private async performDashboardDiscoveryAndSetup(): Promise<void> {
        try {
            console.log('Performing dashboard device discovery...');
            
            // Step 1: List managed things to discover devices
            const hubState = store.getState().hub;
            const hubId = hubState.hubId;
            console.log('Hub ID from store:', hubId);
            console.log('Full hub state:', hubState);
            
            if (!hubId) {
                console.error('No hub ID available for device discovery');
                return;
            }
            
            console.log('Calling deviceService.listManagedThings with hubId:', hubId);
            const deviceIds = await deviceService.listManagedThings(hubId);
            console.log(`ListManagedThings completed. Raw response:`, deviceIds);
            console.log(`Discovered ${deviceIds.length} devices on dashboard:`, deviceIds);
            console.log('Type of deviceIds:', typeof deviceIds, 'Array.isArray:', Array.isArray(deviceIds));
            
            if (deviceIds.length === 0) {
                console.log('No devices found, skipping polling setup');
                return;
            }
            
            // Step 2: For each device, get capabilities and status once
            for (const deviceId of deviceIds) {
                try {
                    console.log(`Getting capabilities and initial status for device ${deviceId}`);
                    
                    // Get device with capabilities (this will cache capabilities for 5 minutes)
                    const device = await deviceService.getDeviceWithDetails(deviceId);
                    console.log(`Device ${deviceId} details retrieved:`, device.Name);
                    
                    // Update Redux store with device info
                    if (device.currentState) {
                        console.log(`Updating Redux store for device ${deviceId}`);
                        store.dispatch(updateDeviceState({ 
                            ManagedThingId: deviceId, 
                            newState: device.currentState 
                        }));
                    } else {
                        console.log(`No current state for device ${deviceId}`);
                    }
                    
                    console.log(`Device ${deviceId} capabilities and status loaded`);
                    
                } catch (error) {
                    console.error(`Error loading device ${deviceId}:`, error);
                }
            }
            
            // Step 3: Start 1-minute ReadState + GetState polling for all devices
            console.log(`Starting 1-minute ReadState+GetState polling for ${deviceIds.length} devices`);
            for (const deviceId of deviceIds) {
                console.log(`Starting dashboard polling for device ${deviceId}`);
                await this.startDashboardDevicePolling(deviceId);
            }
            
            console.log('Dashboard discovery and setup completed');
            
        } catch (error) {
            console.error('Error in dashboard discovery and setup:', error);
            console.error('Error stack:', error instanceof Error ? error.stack : 'No stack trace available');
        }
    }

    /**
     * Start 1-minute ReadState + GetState polling for a single device (dashboard mode)
     * This is different from device-detail polling - no capability calls
     */
    private async startDashboardDevicePolling(deviceId: string): Promise<void> {
        // Don't start if already polling this device
        if (this.getStateIntervals.has(deviceId)) {
            return;
        }
        
        console.log(`Starting dashboard polling for device ${deviceId} (GetState only: 5s, no capabilities)`);
        
        // Set up simplified GetState interval (every 1 minute, no capability calls)
        const getStateInterval = setInterval(() => {
            this.requestQueue.push(async () => {
                try {
                    console.log(`Dashboard simplified GetState cycle for device ${deviceId}`);
                    
                    // Get current state directly from cloud (no ReadState needed)
                    const state = await deviceService.getDeviceState(deviceId);
                    
                    if (state && this.shouldUpdateState(deviceId, state)) {
                        const onOffCapability = state.Endpoints?.[0]?.capabilities?.find(c => c.name === 'On/Off' || c.name === 'OnOff');
                        const onOffValue = onOffCapability?.properties?.find(p => p.name === 'OnOff')?.value?.propertyValue;
                        console.log(`Dashboard REDUX UPDATE - Device: ${deviceId}, OnOff: ${onOffValue}, Timestamp: ${new Date().toISOString()}`);
                        
                        store.dispatch(updateDeviceState({ ManagedThingId: deviceId, newState: state }));

                        const deviceSubscribers = this.subscribers.get(deviceId);
                        if (deviceSubscribers) {
                            deviceSubscribers.forEach(callback => {
                                try {
                                    callback(state);
                                } catch (error) {
                                    console.error(`Error notifying subscriber for device ${deviceId}:`, error);
                                }
                            });
                        }
                    } else if (state) {
                        console.log(`Skipped Dashboard Redux update for device ${deviceId} due to stale state`);
                    }
                    console.log(`Dashboard simplified GetState completed for device ${deviceId}`);
                } catch (error) {
                    console.error(`Error in dashboard simplified GetState cycle for device ${deviceId}:`, error);
                }
            });
            this.processQueue();
        }, this.POLLING_INTERVAL);

        this.getStateIntervals.set(deviceId, getStateInterval);
    }

    /**
     * Start polling only a single device (device detail mode)
     */
    private async startSingleDevicePolling(deviceId: string): Promise<void> {
        console.log(`Starting single device polling for: ${deviceId}`);
        
        // Stop all existing polling first
        this.stopAllDevicePolling();
        
        // Wait a moment to ensure all intervals are cleared
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // Start polling only the focused device
        await this.startPolling(deviceId);
        
        // Log current status
        const currentActiveDevices = [...Array.from(this.readStateIntervals.keys()), ...Array.from(this.getStateIntervals.keys())];
        console.log(`Single device polling active for: ${deviceId}. Active devices:`, currentActiveDevices);
    }

    /**
     * Stop polling all devices
     */
    private stopAllDevicePolling(): void {
        const currentReadStateDevices = Array.from(this.readStateIntervals.keys());
        const currentGetStateDevices = Array.from(this.getStateIntervals.keys());
        const allCurrentDevices = [...currentReadStateDevices, ...currentGetStateDevices];
        console.log(`Stopping all device polling. Currently active: [${allCurrentDevices.join(', ')}]`);
        
        // Stop all ReadState intervals
        this.readStateIntervals.forEach((interval: NodeJS.Timeout, deviceId: string) => {
            clearInterval(interval);
            console.log(`Stopped ReadState polling for device ${deviceId}`);
        });
        this.readStateIntervals.clear();
        
        // Stop all GetState intervals
        this.getStateIntervals.forEach((interval: NodeJS.Timeout, deviceId: string) => {
            clearInterval(interval);
            console.log(`Stopped GetState polling for device ${deviceId}`);
        });
        this.getStateIntervals.clear();
        
        // Clear the request queue to prevent any pending requests
        this.requestQueue = [];
        
        console.log(`All device polling stopped. Active devices now: []`);
    }

    /**
     * Get current polling status
     */
    public getPollingStatus(): { mode: string; focusedDevice: string | null; activeDevices: string[] } {
        const readStateDevices = Array.from(this.readStateIntervals.keys());
        const getStateDevices = Array.from(this.getStateIntervals.keys());
        const activeDevices = [...readStateDevices, ...getStateDevices];
        return {
            mode: this.currentPollingMode,
            focusedDevice: this.focusedDeviceId,
            activeDevices: activeDevices
        };
    }

    public disconnect(): void {
        // Clear all ReadState intervals
        this.readStateIntervals.forEach((interval: NodeJS.Timeout, deviceId: string) => {
            clearInterval(interval);
            console.log(`Stopped ReadState polling for device ${deviceId}`);
        });
        this.readStateIntervals.clear();
        
        // Clear all GetState intervals
        this.getStateIntervals.forEach((interval: NodeJS.Timeout, deviceId: string) => {
            clearInterval(interval);
            console.log(`Stopped GetState polling for device ${deviceId}`);
        });
        this.getStateIntervals.clear();
        
        this.subscribers.clear();
        this.isInitialized = false;
        this.currentPollingMode = 'stopped';
        this.focusedDeviceId = null;
        console.log('DeviceStatePollingService disconnected');
    }
}

// Export singleton instance
export const deviceStatePollingService = DeviceStatePollingService.getInstance();