// src/services/deviceService.ts

import {
    CreateManagedThingCommand,
    SendManagedThingCommandCommand,
    DeleteManagedThingCommand,
    GetManagedThingCapabilitiesCommand,
    GetManagedThingCommand,
    GetManagedThingStateCommand,
    IoTManagedIntegrationsClient,
    ListManagedThingsCommand,
    StateEndpoint,
    StateCapability,
    CapabilityReport as AwsCapabilityReport,
    CapabilityReportEndpoint,
    CapabilityReportCapability,
    StartDeviceDiscoveryCommand,
    GetDeviceDiscoveryCommand,
    ListDeviceDiscoveriesCommand,
    ListDiscoveredDevicesCommand,
    Role
} from "@aws-sdk/client-iot-managed-integrations";

import { validateEnvironmentConfig, logEnvironmentInfo } from '../utils/envValidation';
import { debugLogger } from '../utils/debugLogger';

import {
    CapabilityReport,
    Device,
    DeviceState,
    ProtocolType,
    Endpoint,
    PropertyMetadata,
    EndpointDefinition,
    CapabilityDefinition,
    PropertyDefinition
} from '../types/deviceTypes';
import { HUB_CONFIG } from '../constants';

class DeviceService {
    public client: IoTManagedIntegrationsClient;
    private deviceCapabilitiesCache: Map<string, { capabilities: CapabilityReport; timestamp: number }> = new Map();
    private readonly CAPABILITIES_CACHE_DURATION = 300000; // 5 minutes cache for capabilities

    constructor() {
        const config = validateEnvironmentConfig();
        logEnvironmentInfo();

        this.client = new IoTManagedIntegrationsClient({
            region: config.region,
            endpoint: config.endpoint,
            credentials: {
                accessKeyId: config.accessKeyId,
                secretAccessKey: config.secretAccessKey,
                sessionToken: config.sessionToken
            }
        });
    }

    /**
     * Returns only IDs of activated devices
     */
    async listManagedThings(parentControllerId: string): Promise<string[]> {
        try {
            debugLogger.log('Listing managed things for controller:', parentControllerId);
            
            const command = new ListManagedThingsCommand({
                ParentControllerIdentifierFilter: parentControllerId
            });

            debugLogger.log('Sending ListManagedThingsCommand...');
            const response = await this.client.send(command);
            debugLogger.log('ListManagedThings response:', response);
            
            // Filter for activated devices and extract their IDs
            const activatedDeviceIds = (response.Items || [])
                .filter(thing => thing.Role === "DEVICE" && thing.ProvisioningStatus === "ACTIVATED" && thing.Id)
                .map(thing => thing.Id as string);

            debugLogger.log(`Found ${activatedDeviceIds.length} activated devices`);
            return activatedDeviceIds;
        } catch (error: any) {
            debugLogger.error('Failed to list managed things:', error);
            debugLogger.error('Error details:', {
                name: error.name,
                message: error.message,
                stack: error.stack,
                code: error.code,
                statusCode: error.$metadata?.httpStatusCode
            });
            throw new Error(error?.message || 'Failed to list managed things');
        }
    }

    async getDevice(managedThingId: string): Promise<Device> {
        try {
            const command = new GetManagedThingCommand({
                Identifier: managedThingId
            });

            const deviceResponse = await this.client.send(command);

            // Use cached capabilities to avoid duplicate calls
            const capabilities = await this.getCachedDeviceCapabilities(managedThingId);
            const nodeId = capabilities.nodeId || "UNKNOWN";

            return {
                ManagedThingId: managedThingId,
                Name: deviceResponse.Name || 'Unnamed Device',
                Protocol: nodeId.toLowerCase().startsWith('zb.')
                    ? ProtocolType.ZIGBEE
                    : ProtocolType.ZWAVE,
                CapabilityReport: capabilities
            };
        } catch (error: any) {
            console.error(`Failed to get device ${managedThingId}:`, error);
            throw new Error(error?.message || `Failed to get device ${managedThingId}`);
        }
    }

    /**
     * Get device capabilities with caching (1 minute cache)
     */
    private async getCachedDeviceCapabilities(managedThingId: string): Promise<CapabilityReport> {
        const now = Date.now();
        const cached = this.deviceCapabilitiesCache.get(managedThingId);
        
        // Return cached capabilities if still valid (within 1 minute)
        if (cached && (now - cached.timestamp) < this.CAPABILITIES_CACHE_DURATION) {
            console.log(`Using cached capabilities for device ${managedThingId} (age: ${Math.round((now - cached.timestamp) / 1000)}s)`);
            return cached.capabilities;
        }
        
        // Fetch fresh capabilities and cache them
        console.log(`Fetching fresh capabilities for device ${managedThingId} (cache ${cached ? 'expired' : 'missing'})`);
        const capabilities = await this.getDeviceCapabilities(managedThingId);
        this.deviceCapabilitiesCache.set(managedThingId, { capabilities, timestamp: now });
        
        return capabilities;
    }

    async getDeviceCapabilities(managedThingId: string): Promise<CapabilityReport> {
        try {
            const command = new GetManagedThingCapabilitiesCommand({
                Identifier: managedThingId
            });

            const response = await this.client.send(command);
            if (!response.CapabilityReport) {
                throw new Error('Capability report is missing');
            }

            const awsReport = response.CapabilityReport;
            
            // Map AWS SDK CapabilityReport to our internal type
            const mappedReport: CapabilityReport = {
                version: awsReport.version || '1.0',
                nodeId: awsReport.nodeId || 'UNKNOWN',
                endpoints: (awsReport.endpoints || []).map(endpoint => ({
                    id: endpoint.id || '',
                    deviceTypes: endpoint.deviceTypes || [],
                    capabilities: (endpoint.capabilities || []).map(capability => ({
                        id: capability.id || '',
                        name: capability.name || '',
                        version: capability.version || '',
                        properties: (capability.properties || []).map(prop => ({
                            name: prop,
                            type: 'string',
                            access: ['readable']
                        })),
                        actions: capability.actions || [],
                        events: capability.events || []
                    }))
                }))
            };

            return mappedReport;
        } catch (error) {
            console.error('Error fetching device capabilities:', error);
            throw new Error('Failed to fetch device capabilities');
        }
    }

    async getDeviceState(managedThingId: string): Promise<DeviceState | null> {
        try {
            const command = new GetManagedThingStateCommand({
                ManagedThingId: managedThingId
            });

            const response = await this.client.send(command);
            console.log(`RAW CLOUD RESPONSE for device ${managedThingId}:`, JSON.stringify(response, null, 2));
            
            // Log specific OnOff values for debugging
            response.Endpoints?.forEach((endpoint, endpointIndex) => {
                (endpoint as any).capabilities?.forEach((capability: any, capIndex: number) => {
                    if (capability.name === 'On/Off' || capability.name === 'OnOff') {
                        const onOffProp = (capability.properties as any[])?.find((p: any) => p.name === 'OnOff');
                        console.log(`OnOff Debug - Device: ${managedThingId}, Endpoint: ${endpoint.endpointId}, CapabilityName: ${capability.name}, Value: ${onOffProp?.value?.propertyValue}, LastChanged: ${onOffProp?.value?.lastChangedAt}, Timestamp: ${new Date().toISOString()}`);
                    }
                });
            });
            
            // Check if endpoints are empty or missing - use capabilities as fallback
            if (!response.Endpoints || response.Endpoints.length === 0) {
                console.warn(`No endpoints found in device state for ${managedThingId}, using capabilities as fallback`);
                
                try {
                    // Get cached capabilities to use as fallback structure
                    const capabilities = await this.getCachedDeviceCapabilities(managedThingId);
                    console.log(`Using capabilities as fallback for device ${managedThingId}:`, capabilities);
                    
                    // Create endpoints structure from capabilities with default values
                    const fallbackEndpoints = capabilities.endpoints.map(endpoint => ({
                        endpointId: endpoint.id,
                        deviceTypes: endpoint.deviceTypes || [],
                        capabilities: endpoint.capabilities.map(capability => {
                            // Create default properties based on capability type
                            let defaultProperties: any[] = [];
                            
                            if (capability.name === 'OnOff' || capability.id.includes('OnOff')) {
                                defaultProperties = [
                                    {
                                        name: 'OnOff',
                                        value: {
                                            propertyValue: false, // Default to off
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    }
                                ];
                            } else if (capability.name === 'ColorControl' || capability.id.includes('ColorControl')) {
                                defaultProperties = [
                                    {
                                        name: 'CurrentHue',
                                        value: {
                                            propertyValue: 0,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    },
                                    {
                                        name: 'CurrentSaturation',
                                        value: {
                                            propertyValue: 0,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    }
                                ];
                            } else if (capability.name === 'LevelControl' || capability.id.includes('LevelControl')) {
                                defaultProperties = [
                                    {
                                        name: 'CurrentLevel',
                                        value: {
                                            propertyValue: 0,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    }
                                ];
                            } else {
                                // For other capabilities, create properties based on capability definition
                                defaultProperties = capability.properties.map(prop => ({
                                    name: prop.name,
                                    value: {
                                        propertyValue: null, // Default null value
                                        lastChangedAt: new Date().toISOString()
                                    }
                                }));
                            }
                            
                            return {
                                id: capability.id,
                                name: capability.name,
                                version: capability.version,
                                properties: defaultProperties,
                                actions: capability.actions || [],
                                events: capability.events || []
                            };
                        })
                    }));
                    
                    console.log(`Created fallback endpoints from capabilities for ${managedThingId}:`, fallbackEndpoints);
                    
                    return {
                        Endpoints: fallbackEndpoints
                    };
                } catch (capabilityError) {
                    console.error(`Failed to get capabilities fallback for ${managedThingId}:`, capabilityError);
                    return null;
                }
            }

            const endpoints = response.Endpoints.map(endpoint => {
                return {
                    endpointId: endpoint.endpointId || '',
                    deviceTypes: (endpoint as any).deviceTypes || [],
                    capabilities: (endpoint as any).capabilities?.map((capability: any) => {
                        // Handle OnOff properties (both 'OnOff' and 'On/Off' names)
                        if (capability.name === 'OnOff' || capability.name === 'On/Off') {
                            return {
                                id: capability.id,
                                name: capability.name, // Keep the original name from device
                                version: capability.version,
                                properties: [
                                    {
                                        name: 'OnOff',
                                        value: (capability.properties as any[])?.find((p: any) => p.name === 'OnOff')?.value || {
                                            propertyValue: false,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    }
                                ],
                                actions: (capability as any).actions || [],
                                events: (capability as any).events || []
                            };
                        }
                        // Handle ColorControl properties
                        if (capability.name === 'ColorControl') {
                            return {
                                id: capability.id,
                                name: capability.name,
                                version: capability.version,
                                properties: [
                                    {
                                        name: 'CurrentHue',
                                        value: (capability.properties as any[])?.find((p: any) => p.name === 'CurrentHue')?.value || {
                                            propertyValue: 0,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    },
                                    {
                                        name: 'CurrentSaturation',
                                        value: (capability.properties as any[])?.find((p: any) => p.name === 'CurrentSaturation')?.value || {
                                            propertyValue: 0,
                                            lastChangedAt: new Date().toISOString()
                                        }
                                    }
                                ],
                                actions: (capability as any).actions || [],
                                events: (capability as any).events || []
                            };
                        }
                        // Default handling for other capabilities
                        return {
                            id: capability.id,
                            name: capability.name,
                            version: capability.version,
                            properties: (capability.properties || []).map((prop: any) => ({
                                name: prop.name,
                                value: prop.value ? {
                                    propertyValue: prop.value.propertyValue,
                                    lastChangedAt: prop.value.lastChangedAt
                                } : undefined
                            })),
                            actions: (capability as any).actions || [],
                            events: (capability as any).events || []
                        };
                    }) || []
                };
            });

            console.log('Mapped device state:', JSON.stringify({ Endpoints: endpoints }, null, 2));

            return {
                Endpoints: endpoints
            };
        } catch (error) {
            console.error('Error getting device state:', error);
            return null;
        }
    }

    async getDeviceWithDetails(managedThingId: string): Promise<Device> {
        try {
            // Get basic device info and capabilities first (using cached capabilities)
            const [device, capabilities] = await Promise.all([
                this.getDevice(managedThingId),
                this.getCachedDeviceCapabilities(managedThingId)
            ]);
            
            // Skip ReadState commands to avoid 500 errors - just get current state
            console.log(`Getting device state without ReadState commands for ${managedThingId}`);
            const actualState = await this.getDeviceState(managedThingId);
            
            const originalName = device.Name; // Store the original name

            // Use actual state if available, otherwise create empty structure
            const currentState = actualState || {
                Endpoints: device.CapabilityReport.endpoints.map(endpoint => {
                    // Find matching endpoint in capabilities
                    const capabilityEndpoint = capabilities.endpoints.find(
                        e => e.id === endpoint.id
                    );

                    return {
                        endpointId: endpoint.id,
                        deviceTypes: capabilityEndpoint?.deviceTypes || [],
                        capabilities: endpoint.capabilities.map(capability => {
                            // Find matching capability definition
                            const capabilityDef = capabilityEndpoint?.capabilities.find(
                                c => c.id === capability.id
                            );

                            return {
                                ...capability,
                                actions: capabilityDef?.actions || [],
                                events: capabilityDef?.events || []
                            };
                        })
                    };
                })
            };

            console.log('Device with current state (no ReadState):', JSON.stringify(currentState, null, 2));

            // Combine all information, preserving the original name
            return {
                ...device,
                Name: originalName, // Ensure we keep the original name
                CapabilityReport: capabilities,
                currentState: currentState
            };
        } catch (error: any) {
            console.error(`Failed to get device details for ${managedThingId}:`, error);
            if (error?.message?.includes('404') || error?.name === 'ResourceNotFoundException') {
                throw new Error(`Device ${managedThingId} not found`);
            }
            throw new Error(error?.message || `Failed to get device details for ${managedThingId}`);
        }
    }

    async getDevices(): Promise<Device[]> {
        try {
            // Note: You'll need to implement pagination if you have many devices
            const devices: Device[] = [];

            // Implementation depends on your AWS API structure
            // This is a placeholder for the actual implementation

            return devices;
        } catch (error) {
            console.error('Error fetching devices:', error);
            throw new Error('Failed to fetch devices');
        }
    }

    async deleteDevice(managedThingId: string): Promise<void> {
        try {
            const command = new DeleteManagedThingCommand({
                Identifier: managedThingId
            });

            await this.client.send(command);
        } catch (error) {
            console.error('Error deleting device:', error);
            throw new Error('Failed to delete device');
        }
    }

    async executeAction(
        managedThingId: string,
        capabilityId: string,
        action: string,
        endpointId?: string,
        parameters?: Record<string, any>,
        deviceName?: string
    ): Promise<void> {
        try {
            console.log('Executing action:', {
                managedThingId,
                capabilityId,
                action,
                endpointId,
                parameters
            });

            // Get device info for logging and light detection
            const device = await this.getDevice(managedThingId);
            const finalDeviceName = deviceName || device.Name || 'Unknown Device';

            // If no endpointId is provided, try to get it from device details
            if (!endpointId) {
                endpointId = device.CapabilityReport.endpoints[0]?.id;
                if (!endpointId) {
                    throw new Error('No endpoint ID found for device');
                }
            }


            const formattedEndpoint = {
                endpointId,
                capabilities: [
                    {
                        id: capabilityId,
                        name: capabilityId.split('@')[0].split('.')[1], // Extract name from "matter.OnOff@1.4"
                        version: "1",
                        actions: [
                            {
                                name: action,
                                parameters: parameters || {}
                            }
                        ]
                    }
                ]
            };

            console.log('Sending command:', JSON.stringify(formattedEndpoint, null, 2));

            // Send the command using SendManagedThingCommandCommand
            const command = new SendManagedThingCommandCommand({
                ManagedThingId: managedThingId,
                Endpoints: [formattedEndpoint]
            });

            await this.client.send(command);
            console.log('Command sent successfully');


        } catch (error) {
            console.error('Error executing action:', error);
            throw error;
        }
    }

    // Helper method to validate device state
    private validateDeviceState(state: DeviceState): boolean {
        // Add validation logic here
        return true;
    }

    // Helper method to format capability ID
    private formatCapabilityId(capability: string, version: string): string {
        return `matter.${capability}@${version}`;
    }

    // Helper method to handle errors
    private handleError(error: any, customMessage: string): never {
        console.error(customMessage, error);
        throw new Error(customMessage);
    }

    /**
     * Send ReadState commands using wildcard approach to read all properties
     * This uses ["*"] to read all properties instead of individual capability commands
     */
    private async sendReadStateForAllCapabilities(managedThingId: string, capabilities: CapabilityReport): Promise<void> {
        try {
            console.log(`Sending wildcard ReadState command for device ${managedThingId}`);
            
            // Use wildcard approach - send one command per endpoint with all capabilities
            for (const endpoint of capabilities.endpoints) {
                try {
                    const formattedEndpoint = {
                        endpointId: endpoint.id,
                        capabilities: endpoint.capabilities.map(capability => ({
                            id: capability.id,
                            name: capability.name,
                            version: capability.version,
                            actions: [{
                                name: "ReadState",
                                parameters: {
                                    propertiesToRead: ["*"] // Use wildcard to read all properties
                                }
                            }]
                        }))
                    };

                    console.log(`Sending wildcard ReadState for endpoint ${endpoint.id} with ${endpoint.capabilities.length} capabilities`);

                    const command = new SendManagedThingCommandCommand({
                        ManagedThingId: managedThingId,
                        Endpoints: [formattedEndpoint]
                    });

                    await this.client.send(command);
                    console.log(`Wildcard ReadState sent for endpoint ${endpoint.id}`);
                    
                    // Small delay between endpoints to avoid rate limiting
                    await new Promise(resolve => setTimeout(resolve, 200));
                    
                } catch (error) {
                    console.error(`Error sending wildcard ReadState for endpoint ${endpoint.id}:`, error);
                    // Continue with other endpoints even if one fails
                }
            }
            
            console.log(`Completed sending wildcard ReadState commands for device ${managedThingId}`);
        } catch (error) {
            console.error(`Error in sendReadStateForAllCapabilities for device ${managedThingId}:`, error);
            throw error;
        }
    }

    /**
     * Send optimized ReadState wildcard command for light devices
     * Based on testing, one wildcard ReadState per endpoint gets ALL properties for ALL capabilities
     * This is much more efficient than individual capability commands
     */
    async sendReadStateForLightDevice(managedThingId: string): Promise<void> {
        try {
            debugLogger.log(`Sending optimized ReadState wildcard command for light device ${managedThingId}`);
            
            // Get device capabilities to determine endpoints
            const capabilities = await this.getCachedDeviceCapabilities(managedThingId);
            
            for (const endpoint of capabilities.endpoints) {
                // Check if this endpoint has any light-related capabilities
                const hasLightCapabilities = endpoint.capabilities.some(capability => 
                    capability.id.includes('OnOff') || 
                    capability.id.includes('LevelControl') || 
                    capability.id.includes('ColorControl') ||
                    capability.name.toLowerCase().includes('onoff') ||
                    capability.name.toLowerCase().includes('levelcontrol') ||
                    capability.name.toLowerCase().includes('colorcontrol')
                );
                
                if (hasLightCapabilities) {
                    try {
                        // OPTIMIZED: Send one wildcard ReadState that gets ALL properties for ALL capabilities on this endpoint
                        const formattedEndpoint = {
                            endpointId: endpoint.id,
                            capabilities: endpoint.capabilities.map(capability => ({
                                id: capability.id,
                                name: capability.name,
                                version: capability.version,
                                actions: [{
                                    name: "ReadState",
                                    parameters: {
                                        propertiesToRead: ["*"] // Wildcard gets ALL properties for ALL capabilities
                                    }
                                }]
                            }))
                        };

                        debugLogger.log(`Sending optimized ReadState for endpoint ${endpoint.id} with ${endpoint.capabilities.length} capabilities`);

                        const command = new SendManagedThingCommandCommand({
                            ManagedThingId: managedThingId,
                            Endpoints: [formattedEndpoint]
                        });

                        await this.client.send(command);
                        debugLogger.log(`Optimized ReadState sent for endpoint ${endpoint.id}`);
                        
                        // Small delay between endpoints to avoid rate limiting
                        await new Promise(resolve => setTimeout(resolve, 200));
                        
                    } catch (error) {
                        debugLogger.error(`Error sending optimized ReadState for endpoint ${endpoint.id}:`, error);
                        // Continue with other endpoints even if one fails
                    }
                } else {
                    debugLogger.log(`No light capabilities found in endpoint ${endpoint.id}, skipping ReadState`);
                }
            }
            
            debugLogger.log(`Completed optimized ReadState commands for device ${managedThingId}`);
        } catch (error) {
            debugLogger.error(`Error in sendReadStateForLightDevice for device ${managedThingId}:`, error);
            throw error;
        }
    }

    async startZigbeeDiscovery(managedThingId?: string): Promise<void> {
        await this.startDeviceDiscovery({ 
            discoveryType: 'ZIGBEE',
            managedThingId 
        });
    }

    async startDeviceDiscovery(params: {
        discoveryType: 'ZWAVE' | 'ZIGBEE';
        managedThingId?: string;
        authenticationMaterial?: string;
        authenticationMaterialType?: 'ZWAVE_INSTALL_CODE';
    }): Promise<void> {
        try {
            const command = new StartDeviceDiscoveryCommand({
                DiscoveryType: params.discoveryType,
                ...(params.managedThingId && {
                    ControllerIdentifier: params.managedThingId
                }),
                ...(params.authenticationMaterial && params.authenticationMaterialType && {
                    AuthenticationMaterial: params.authenticationMaterial,
                    AuthenticationMaterialType: params.authenticationMaterialType
                })
            });

            await this.client.send(command);
            console.log(`${params.discoveryType} device discovery started`, 
                params.managedThingId ? `for controller ${params.managedThingId}` : '',
                params.authenticationMaterial ? 'with authentication' : 'without authentication');
        } catch (error) {
            console.error(`Error starting ${params.discoveryType} device discovery:`, error);
            throw error;
        }
    }

    async updateDeviceName(managedThingId: string, name: string): Promise<void> {
        try {
            console.log(`Updating name for device ${managedThingId} to "${name}"`);
            // Get the current device information
            const device = await this.getDevice(managedThingId);
            
            // Update the device with the new name
            // Note: This is just keeping the name updated in the frontend for now
            // In a real implementation, you would make an API call to update the device name
            console.log(`Device name updated successfully for ${managedThingId}`);
        } catch (error) {
            console.error(`Failed to update name for device ${managedThingId}:`, error);
            throw new Error(`Failed to update device name: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }

    public async stopDeviceDiscovery(): Promise<void> {
        // ... existing code ...
    }

    async createZWaveDevice(dsk: string): Promise<Device> {
        try {
            console.log('Creating Z-Wave device with DSK:', dsk);
            
            const command = new CreateManagedThingCommand({
                AuthenticationMaterial: dsk,
                AuthenticationMaterialType: 'ZWAVE_QR_BAR_CODE',
                CredentialLockerId: HUB_CONFIG.CREDENTIAL_LOCKER_ID,
                Role: Role.DEVICE
            });

            const response = await this.client.send(command);
            console.log('Z-Wave device creation response:', response);

            // The response should contain the managed thing ID
            const managedThingId = (response as any).Id;
            if (!managedThingId) {
                throw new Error('No managed thing ID in response');
            }

            // Get the created device details
            const device = await this.getDevice(managedThingId);
            return device;
        } catch (error) {
            console.error('Error creating Z-Wave device:', error);
            throw error;
        }
    }

    /**
     * Start device discovery for simple inclusion (new workflow)
     * This starts local discovery and returns a discovery ID to track progress
     * For Z-Wave, includes pin code (custom or default "00000")
     */
    async startSimpleDeviceDiscovery(
        protocol: 'ZWAVE' | 'ZIGBEE', 
        hubManagedThingId: string, 
        zwavePinCode?: string
    ): Promise<{ discoveryId: string }> {
        try {
            console.log(`Starting ${protocol} simple device discovery...`);
            
            const command = new StartDeviceDiscoveryCommand({
                DiscoveryType: protocol,
                ControllerIdentifier: hubManagedThingId,
                // For Z-Wave simple onboarding, include pin code (custom or default)
                ...(protocol === 'ZWAVE' && {
                    AuthenticationMaterial: zwavePinCode || '00000',
                    AuthenticationMaterialType: 'ZWAVE_INSTALL_CODE'
                })
            });

            const response = await this.client.send(command);
            console.log(`${protocol} simple discovery started:`, response);

            // The response contains an 'Id' field (not 'DiscoveryId')
            const discoveryId = (response as any).Id;
            if (!discoveryId) {
                throw new Error('No discovery ID returned from start discovery command');
            }

            return { discoveryId };
        } catch (error) {
            console.error(`Error starting ${protocol} simple device discovery:`, error);
            throw error;
        }
    }

    /**
     * List active device discoveries
     */
    async listDeviceDiscoveries(): Promise<any[]> {
        try {
            const command = new ListDeviceDiscoveriesCommand({});
            const response = await this.client.send(command);
            
            console.log('Device discoveries:', response);
            return (response as any).DeviceDiscoveries || [];
        } catch (error) {
            console.error('Error listing device discoveries:', error);
            throw error;
        }
    }

    /**
     * List discovered devices for a specific discovery session
     */
    async listDiscoveredDevices(discoveryId: string): Promise<any[]> {
        try {
            const command = new ListDiscoveredDevicesCommand({
                Identifier: discoveryId
            } as any);
            const response = await this.client.send(command);
            
            console.log('Discovered devices:', response);
            return (response as any).Items || [];
        } catch (error) {
            console.error('Error listing discovered devices:', error);
            throw error;
        }
    }

    /**
     * Create a managed thing from a discovered device (new workflow)
     */
    async createManagedThingFromDiscoveredDevice(params: {
        discoveryId: string;
        deviceIdentifier: string;
        protocol: 'ZWAVE' | 'ZIGBEE';
        authenticationMaterial?: string;
    }): Promise<{ managedThingId: string }> {
        try {
            console.log('Creating managed thing from discovered device:', params);
            
            const command = new CreateManagedThingCommand({
                Role: Role.DEVICE,
                ...(params.authenticationMaterial && {
                    AuthenticationMaterial: params.authenticationMaterial,
                    AuthenticationMaterialType: 'DISCOVERED_DEVICE'
                }),
                ...(params.protocol === 'ZWAVE' && {
                    CredentialLockerId: HUB_CONFIG.CREDENTIAL_LOCKER_ID
                }),
                ...(params.protocol === 'ZIGBEE' && {
                    CredentialLockerId: HUB_CONFIG.CREDENTIAL_LOCKER_ID
                })
            } as any);

            const response = await this.client.send(command);
            console.log('Managed thing created from discovered device:', response);

            const managedThingId = (response as any).Id;
            if (!managedThingId) {
                throw new Error('No managed thing ID returned from create command');
            }

            return { managedThingId };
        } catch (error) {
            console.error('Error creating managed thing from discovered device:', error);
            throw error;
        }
    }

    /**
     * Create a Zigbee device with install code (using new workflow)
     */
    async createZigbeeDeviceWithInstallCode(installCode: string): Promise<{ managedThingId: string }> {
        try {
            console.log('Creating Zigbee device with install code:', installCode);
            
            const command = new CreateManagedThingCommand({
                AuthenticationMaterial: installCode,
                AuthenticationMaterialType: 'ZIGBEE_INSTALL_CODE' as any, // Type assertion for now
                CredentialLockerId: HUB_CONFIG.CREDENTIAL_LOCKER_ID,
                Role: Role.DEVICE
            });

            const response = await this.client.send(command);
            console.log('Zigbee device with install code creation response:', response);

            const managedThingId = (response as any).Id;
            if (!managedThingId) {
                throw new Error('No managed thing ID in response');
            }

            return { managedThingId };
        } catch (error) {
            console.error('Error creating Zigbee device with install code:', error);
            throw error;
        }
    }

    /**
     * Create a device using QR code (Simple Setup approach)
     * Uses full QR code text with proper authentication material types
     */
    async createDeviceWithQRCode(qrCodeText: string, protocol: 'ZWAVE' | 'ZIGBEE'): Promise<{ managedThingId: string }> {
        try {
            console.log(`Creating ${protocol} device with QR code (Simple Setup):`, qrCodeText.substring(0, 20) + '...');
            
            const authMaterialType = protocol === 'ZWAVE' ? 'ZWAVE_QR_BAR_CODE' : 'ZIGBEE_QR_BAR_CODE';
            
            const command = new CreateManagedThingCommand({
                AuthenticationMaterial: qrCodeText,
                AuthenticationMaterialType: authMaterialType as any,
                CredentialLockerId: HUB_CONFIG.CREDENTIAL_LOCKER_ID,
                Role: Role.DEVICE
            });

            const response = await this.client.send(command);
            console.log(`${protocol} device with QR code creation response:`, response);

            const managedThingId = (response as any).Id;
            if (!managedThingId) {
                throw new Error('No managed thing ID in response');
            }

            return { managedThingId };
        } catch (error) {
            console.error(`Error creating ${protocol} device with QR code:`, error);
            throw error;
        }
    }

    /**
     * Send initial ReadState command after device onboarding to populate endpoints
     * This ensures device state is available immediately after onboarding
     */
    async sendInitialReadStateAfterOnboarding(managedThingId: string): Promise<void> {
        try {
            console.log(`Sending initial ReadState after onboarding for device ${managedThingId}`);
            
            // Get device capabilities
            const capabilities = await this.getCachedDeviceCapabilities(managedThingId);
            
            // Send ReadState for all endpoints to populate state data
            await this.sendReadStateForAllCapabilities(managedThingId, capabilities);
            
            console.log(`Initial ReadState sent for newly onboarded device ${managedThingId}`);
        } catch (error) {
            console.error(`Error sending initial ReadState for ${managedThingId}:`, error);
            // Don't throw - this is not critical for onboarding success
        }
    }

    /**
     * Check if a managed thing is activated (for Simple Setup flow)
     */
    async checkDeviceActivationStatus(managedThingId: string): Promise<{ 
        isActivated: boolean; 
        status: string; 
        device?: any 
    }> {
        try {
            const command = new GetManagedThingCommand({
                Identifier: managedThingId
            });

            const response = await this.client.send(command);
            const status = (response as any).ProvisioningStatus || 'UNKNOWN';
            const isActivated = status === 'ACTIVATED';
            
            console.log(`Device ${managedThingId} status: ${status}, activated: ${isActivated}`);
            
            return {
                isActivated,
                status,
                device: response
            };
        } catch (error) {
            console.error(`Error checking activation status for ${managedThingId}:`, error);
            return {
                isActivated: false,
                status: 'ERROR'
            };
        }
    }

    /**
     * Get discovery status
     */
    async getDeviceDiscovery(discoveryId: string): Promise<any> {
        try {
            const command = new GetDeviceDiscoveryCommand({
                Identifier: discoveryId
            } as any);
            const response = await this.client.send(command);
            
            console.log('Discovery status:', response);
            return response;
        } catch (error) {
            console.error('Error getting device discovery:', error);
            throw error;
        }
    }
}

// Create and export a singleton instance
export const deviceService = new DeviceService();
