import { deviceService } from './deviceService';
import { store } from '../store/store';
import { addDevice, removeDevice, updateDeviceState } from '../store/deviceSlice';
import { deviceStatePollingService } from './deviceStatePollingService';
import { HUB_CONFIG } from '../constants';

class DeviceMonitorService {
    private static instance: DeviceMonitorService | null = null;
    private pollingInterval: NodeJS.Timeout | null = null;
    private knownDevices: Set<string> = new Set();
    private capabilityRefreshTimeouts: Map<string, NodeJS.Timeout> = new Map();
    private currentMode: 'dashboard' | 'device-detail' | 'onboarding' | 'stopped' = 'stopped';

    private constructor() {
        console.log('DeviceMonitorService constructed');
    }

    public static getInstance(): DeviceMonitorService {
        if (!DeviceMonitorService.instance) {
            DeviceMonitorService.instance = new DeviceMonitorService();
        }
        return DeviceMonitorService.instance;
    }

    public async startMonitoring(): Promise<void> {
        if (this.pollingInterval) {
            console.log('Device monitoring is already running');
            return;
        }

        console.log('Starting device monitoring...');
        
        // Initial poll
        await this.pollDevices();

        // Set up device discovery polling interval
        this.pollingInterval = setInterval(async () => {
            await this.pollDevices();
        }, 60000); // Poll every 60 seconds for device discovery

        console.log('Device discovery monitoring started. State polling handled by DeviceStatePollingService.');
    }

    public stopMonitoring(): void {
        if (this.pollingInterval) {
            clearInterval(this.pollingInterval);
            this.pollingInterval = null;
        }
        console.log('Stopped device monitoring');
    }

    /**
     * Set monitoring mode based on current page/screen
     */
    public setMonitoringMode(mode: 'dashboard' | 'device-detail' | 'onboarding' | 'stopped'): void {
        console.log(`DeviceMonitorService setting mode to: ${mode}`);
        
        this.currentMode = mode;
        
        switch (mode) {
            case 'dashboard':
                // Start device discovery monitoring for dashboard
                this.startMonitoring();
                break;
                
            case 'device-detail':
                // Stop device discovery monitoring when viewing single device
                this.stopMonitoring();
                break;
                
            case 'onboarding':
                // Keep device discovery monitoring during onboarding to detect new devices
                this.startMonitoring();
                break;
                
            case 'stopped':
                // Stop all monitoring
                this.stopMonitoring();
                break;
        }
    }

    private async pollDevices(): Promise<void> {
        try {
            const currentDeviceIds = await deviceService.listManagedThings(HUB_CONFIG.MANAGED_THING_ID);
            
            // Find new devices
            const newDevices = currentDeviceIds.filter(id => !this.knownDevices.has(id));
            
            // Find removed devices
            const removedDevices = Array.from(this.knownDevices).filter(id => !currentDeviceIds.includes(id));

            // Update known devices set
            this.knownDevices = new Set(currentDeviceIds);

            // Handle new devices
            for (const deviceId of newDevices) {
                try {
                    const device = await deviceService.getDeviceWithDetails(deviceId);
                    if (device) {
                        store.dispatch(addDevice(device));
                        console.log(`Added new device: ${deviceId}`, device.CapabilityReport.endpoints[0]?.deviceTypes[0] || 'Unknown Device');
                        
                        // Don't automatically start polling - let page-based control handle it
                        console.log(`New device ${deviceId} detected - polling controlled by page mode`);
                        
                        // If this is an "Unknown Device", schedule a capability refresh
                        if (this.isUnknownDevice(device)) {
                            console.log(`Device ${deviceId} is unknown, scheduling capability refresh...`);
                            this.scheduleCapabilityRefresh(deviceId);
                        }
                    }
                } catch (error) {
                    console.error(`Error fetching details for new device ${deviceId}:`, error);
                }
            }

            // Handle removed devices
            for (const deviceId of removedDevices) {
                store.dispatch(removeDevice(deviceId));
                console.log(`Removed device: ${deviceId}`);
                
                // Stop state polling for the removed device
                deviceStatePollingService.stopPolling(deviceId);
            }
        } catch (error) {
            console.error('Error polling devices:', error);
        }
    }

    private isUnknownDevice(device: any): boolean {
        // Check if device has no capabilities or is marked as "Unknown Device"
        const hasNoCapabilities = !device.CapabilityReport?.endpoints?.[0]?.capabilities?.length;
        const isUnknownType = device.CapabilityReport?.endpoints?.[0]?.deviceTypes?.[0] === 'Unknown Device' || 
                             !device.CapabilityReport?.endpoints?.[0]?.deviceTypes?.length;
        
        console.log(`Checking if device ${device.ManagedThingId} is unknown:`, {
            hasNoCapabilities,
            isUnknownType,
            deviceTypes: device.CapabilityReport?.endpoints?.[0]?.deviceTypes,
            capabilitiesCount: device.CapabilityReport?.endpoints?.[0]?.capabilities?.length || 0,
            capabilities: device.CapabilityReport?.endpoints?.[0]?.capabilities?.map((c: any) => c.name) || []
        });
        
        return hasNoCapabilities || isUnknownType;
    }

    private scheduleCapabilityRefresh(deviceId: string): void {
        // Clear any existing timeout for this device
        const existingTimeout = this.capabilityRefreshTimeouts.get(deviceId);
        if (existingTimeout) {
            clearTimeout(existingTimeout);
        }

        // Schedule capability refresh in 10 seconds, then every 30 seconds up to 5 times
        let attempts = 0;
        const maxAttempts = 5;
        
        const refreshCapabilities = async () => {
            attempts++;
            console.log(`Attempting capability refresh for device ${deviceId} (attempt ${attempts}/${maxAttempts})`);
            
            try {
                const updatedDevice = await deviceService.getDeviceWithDetails(deviceId);
                
                if (!this.isUnknownDevice(updatedDevice)) {
                    console.log(`Device ${deviceId} capabilities updated successfully!`);
                    store.dispatch(addDevice(updatedDevice));
                    this.capabilityRefreshTimeouts.delete(deviceId);
                    return;
                }
                
                if (attempts < maxAttempts) {
                    console.log(`Device ${deviceId} still unknown, scheduling next refresh...`);
                    const timeout = setTimeout(refreshCapabilities, 30000); // 30 seconds
                    this.capabilityRefreshTimeouts.set(deviceId, timeout);
                } else {
                    console.log(`Max refresh attempts reached for device ${deviceId}`);
                    this.capabilityRefreshTimeouts.delete(deviceId);
                }
            } catch (error) {
                console.error(`Error refreshing capabilities for device ${deviceId}:`, error);
                if (attempts < maxAttempts) {
                    const timeout = setTimeout(refreshCapabilities, 30000);
                    this.capabilityRefreshTimeouts.set(deviceId, timeout);
                } else {
                    this.capabilityRefreshTimeouts.delete(deviceId);
                }
            }
        };

        // Start first refresh in 10 seconds
        const timeout = setTimeout(refreshCapabilities, 10000);
        this.capabilityRefreshTimeouts.set(deviceId, timeout);
    }
}

export const deviceMonitorService = DeviceMonitorService.getInstance();