// src/services/customEndpointService.ts

import {
    GetCustomEndpointCommand,
    IoTManagedIntegrationsClient,
    GetCustomEndpointCommandOutput
} from "@aws-sdk/client-iot-managed-integrations";

export interface CustomEndpointConfig {
    region: string;
    endpoint?: string;
    credentials: {
        accessKeyId: string;
        secretAccessKey: string;
        sessionToken?: string;
    };
}

export interface CustomEndpointResult {
    customEndpoint?: any; // Using any since CustomEndpoint type is not exported
    success: boolean;
    error?: string;
}

class CustomEndpointService {
    private client: IoTManagedIntegrationsClient;
    private config: CustomEndpointConfig;

    constructor(config: CustomEndpointConfig) {
        this.config = config;
        this.client = new IoTManagedIntegrationsClient({
            region: config.region,
            endpoint: config.endpoint,
            credentials: config.credentials
        });
    }

    /**
     * Update the client configuration (useful for developer mode)
     */
    updateConfig(newConfig: CustomEndpointConfig): void {
        this.config = newConfig;
        this.client = new IoTManagedIntegrationsClient({
            region: newConfig.region,
            endpoint: newConfig.endpoint,
            credentials: newConfig.credentials
        });
    }

    /**
     * Get custom endpoint using the GetCustomEndpointCommand
     * @param customEndpointId - The ID of the custom endpoint to retrieve
     */
    async getCustomEndpoint(customEndpointId: string): Promise<CustomEndpointResult> {
        try {
            console.log(`Getting custom endpoint with ID: ${customEndpointId}`);
            console.log(`Using region: ${this.config.region}`);
            console.log(`Using endpoint: ${this.config.endpoint || 'default'}`);

            const command = new GetCustomEndpointCommand({
                Identifier: customEndpointId
            });

            const response = await this.client.send(command);
            
            // Log the response to understand its structure
            console.log('Custom endpoint response:', response);

            return {
                customEndpoint: (response as any).CustomEndpoint || response,
                success: true
            };
        } catch (error: any) {
            console.error('Failed to get custom endpoint:', error);
            
            return {
                success: false,
                error: error?.message || 'Failed to get custom endpoint'
            };
        }
    }

    /**
     * Get current configuration (useful for debugging)
     */
    getCurrentConfig(): CustomEndpointConfig {
        return { ...this.config };
    }

    /**
     * Test the connection with current configuration
     */
    async testConnection(): Promise<{ success: boolean; error?: string }> {
        try {
            // Try to make a simple call to test the connection
            // We'll use a dummy ID to test the connection - the error response will tell us if we can connect
            const command = new GetCustomEndpointCommand({
                Identifier: "test-connection-dummy-id"
            });

            await this.client.send(command);
            
            return { success: true };
        } catch (error: any) {
            // If we get a specific AWS error (not a network error), the connection is working
            if (error.name === 'ResourceNotFoundException' || 
                error.name === 'ValidationException' ||
                error.$metadata?.httpStatusCode) {
                return { success: true };
            }
            
            return {
                success: false,
                error: error?.message || 'Connection test failed'
            };
        }
    }
}

export default CustomEndpointService;
