import React, { useState } from 'react';
import {
    Box,
    Card,
    CardContent,
    Typography,
    Button,
    Container,
    Grid,
    Chip
} from '@mui/material';
import {
    Router as ZWaveIcon,
    Wifi as ZigbeeIcon,
    ArrowBack,
    Speed as SimpleIcon,
    QrCode as QrCodeIcon
} from '@mui/icons-material';
import { useNavigate } from 'react-router-dom';
import { useDispatch } from 'react-redux';
import { ProtocolType } from '../types/deviceTypes';
import SimpleDeviceOnboarding from '../components/onboarding/SimpleDeviceOnboarding';
import QRCodeOnboarding from '../components/onboarding/QRCodeOnboarding';
import { deviceService } from '../services/deviceService';
import { addDevice, updateDeviceName } from '../store/deviceSlice';
import { deviceStatePollingService } from '../services/deviceStatePollingService';

enum OnboardingStep {
    PROTOCOL_SELECTION = 0,
    METHOD_SELECTION = 1,
    EXECUTION = 2
}

enum OnboardingMethod {
    SIMPLE_ADDITION = 'simple',
    QR_CODE_SCANNER = 'qr',
    MANUAL_CODE_ENTRY = 'manual'
}

const SimpleOnboarding: React.FC = () => {
    const [currentStep, setCurrentStep] = useState<OnboardingStep>(OnboardingStep.PROTOCOL_SELECTION);
    const [selectedProtocol, setSelectedProtocol] = useState<ProtocolType | null>(null);
    const [selectedMethod, setSelectedMethod] = useState<OnboardingMethod | null>(null);
    const navigate = useNavigate();
    const dispatch = useDispatch();

    // Set polling mode to onboarding when component mounts
    React.useEffect(() => {
        console.log('SimpleOnboarding mounted - setting polling mode to onboarding');
        deviceStatePollingService.setPollingMode('onboarding');
        
        return () => {
            console.log('SimpleOnboarding unmounting - stopping polling');
            deviceStatePollingService.setPollingMode('stopped');
        };
    }, []);

    const handleProtocolSelect = (protocol: ProtocolType) => {
        setSelectedProtocol(protocol);
        setCurrentStep(OnboardingStep.METHOD_SELECTION);
    };

    const handleMethodSelect = (method: OnboardingMethod) => {
        setSelectedMethod(method);
        setCurrentStep(OnboardingStep.EXECUTION);
    };

    const handleOnboardingComplete = async (managedThingId: string, deviceName?: string) => {
        console.log('Onboarding completed for device:', managedThingId, 'with name:', deviceName);
        
        try {
            // Get the full device details and add it to the store
            const deviceWithDetails = await deviceService.getDeviceWithDetails(managedThingId);
            
            // If a device name was provided, use it
            if (deviceName) {
                console.log(`Setting device name: ${managedThingId} -> "${deviceName}"`);
                deviceWithDetails.Name = deviceName;
            }
            
            // Add the complete device to Redux store
            console.log('Adding device to Redux store:', deviceWithDetails);
            dispatch(addDevice(deviceWithDetails));
            
            // Also update the name specifically if provided
            if (deviceName) {
                dispatch(updateDeviceName({
                    ManagedThingId: managedThingId,
                    name: deviceName
                }));
            }
            
            console.log('Device successfully added to store, navigating to dashboard');
            
            // Navigate to dashboard to see the named device
            navigate('/');
        } catch (error) {
            console.error('Error completing onboarding:', error);
            // Still navigate even if there's an error
            navigate('/');
        }
    };

    const handleCancel = () => {
        if (currentStep === OnboardingStep.EXECUTION) {
            setCurrentStep(OnboardingStep.METHOD_SELECTION);
            setSelectedMethod(null);
        } else if (currentStep === OnboardingStep.METHOD_SELECTION) {
            setCurrentStep(OnboardingStep.PROTOCOL_SELECTION);
            setSelectedProtocol(null);
        } else {
            navigate('/');
        }
    };

    const renderProtocolSelection = () => (
        <Container maxWidth="md">
            <Box sx={{ textAlign: 'center', mb: 4 }}>
                <Typography variant="h3" gutterBottom>
                    Device Onboarding
                </Typography>
                <Typography variant="h6" color="text.secondary" sx={{ mb: 4 }}>
                    Choose your device protocol to get started
                </Typography>
            </Box>

            <Grid container spacing={3} justifyContent="center">
                <Grid size={{ xs: 12, md: 6 }}>
                    <Card 
                        sx={{ 
                            cursor: 'pointer', 
                            height: '100%',
                            transition: 'all 0.2s',
                            '&:hover': { 
                                transform: 'translateY(-4px)',
                                boxShadow: 4
                            }
                        }}
                        onClick={() => handleProtocolSelect(ProtocolType.ZWAVE)}
                    >
                        <CardContent sx={{ textAlign: 'center', p: 4 }}>
                            <ZWaveIcon sx={{ fontSize: 64, color: 'primary.main', mb: 2 }} />
                            <Typography variant="h5" gutterBottom>
                                Z-Wave Device
                            </Typography>
                            <Typography variant="body1" color="text.secondary" sx={{ mb: 2 }}>
                                Add Z-Wave devices with multiple security options
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                • Simple inclusion<br/>
                                • QR code scanning<br/>
                                • Manual DSK entry
                            </Typography>
                        </CardContent>
                    </Card>
                </Grid>

                <Grid size={{ xs: 12, md: 6 }}>
                    <Card 
                        sx={{ 
                            cursor: 'pointer', 
                            height: '100%',
                            transition: 'all 0.2s',
                            '&:hover': { 
                                transform: 'translateY(-4px)',
                                boxShadow: 4
                            }
                        }}
                        onClick={() => handleProtocolSelect(ProtocolType.ZIGBEE)}
                    >
                        <CardContent sx={{ textAlign: 'center', p: 4 }}>
                            <ZigbeeIcon sx={{ fontSize: 64, color: 'secondary.main', mb: 2 }} />
                            <Typography variant="h5" gutterBottom>
                                Zigbee Device
                            </Typography>
                            <Typography variant="body1" color="text.secondary" sx={{ mb: 2 }}>
                                Add Zigbee devices with flexible options
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                • Simple inclusion<br/>
                                • QR code scanning<br/>
                                • Compatible with Zigbee 3.0
                            </Typography>
                        </CardContent>
                    </Card>
                </Grid>
            </Grid>

            <Box sx={{ textAlign: 'center', mt: 4 }}>
                <Button
                    variant="outlined"
                    startIcon={<ArrowBack />}
                    onClick={() => navigate('/')}
                    size="large"
                >
                    Back to Dashboard
                </Button>
            </Box>
        </Container>
    );

    const renderMethodSelection = () => (
        <Container maxWidth="lg">
            <Box sx={{ textAlign: 'center', mb: 4 }}>
                <Typography variant="h3" gutterBottom>
                    Choose Onboarding Method
                </Typography>
                <Typography variant="h6" color="text.secondary" sx={{ mb: 2 }}>
                    Select how you want to add your {selectedProtocol} device
                </Typography>
                <Chip 
                    label={`${selectedProtocol} Device`} 
                    color="primary" 
                    sx={{ mb: 3 }}
                />
            </Box>

            <Grid container spacing={3} justifyContent="center">
                {/* User Guided Setup */}
                <Grid size={{ xs: 12, md: 6 }}>
                    <Card 
                        sx={{ 
                            cursor: 'pointer', 
                            height: '100%',
                            transition: 'all 0.2s',
                            '&:hover': { 
                                transform: 'translateY(-4px)',
                                boxShadow: 4
                            }
                        }}
                        onClick={() => handleMethodSelect(OnboardingMethod.SIMPLE_ADDITION)}
                    >
                        <CardContent sx={{ textAlign: 'center', p: 4 }}>
                            <SimpleIcon sx={{ fontSize: 64, color: 'success.main', mb: 2 }} />
                            <Typography variant="h6" gutterBottom>
                                User Guided Setup
                            </Typography>
                            <Chip label="Step-by-Step" color="success" size="small" sx={{ mb: 2 }} />
                            <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                {selectedProtocol === ProtocolType.ZWAVE 
                                    ? 'Guided device discovery with optional security codes'
                                    : 'Guided device discovery and setup'
                                }
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                {selectedProtocol === ProtocolType.ZWAVE ? (
                                    <>
                                        • Start device discovery<br/>
                                        • Optional authentication material<br/>
                                        • Create managed thing<br/>
                                        • Guided setup process
                                    </>
                                ) : (
                                    <>
                                        • Start device discovery<br/>
                                        • Create managed thing<br/>
                                        • Guided setup process<br/>
                                        • Device activation
                                    </>
                                )}
                            </Typography>
                        </CardContent>
                    </Card>
                </Grid>

                {/* QR Code Scanner */}
                <Grid size={{ xs: 12, md: 6 }}>
                    <Card 
                        sx={{ 
                            cursor: 'pointer', 
                            height: '100%',
                            transition: 'all 0.2s',
                            '&:hover': { 
                                transform: 'translateY(-4px)',
                                boxShadow: 4
                            }
                        }}
                        onClick={() => handleMethodSelect(OnboardingMethod.QR_CODE_SCANNER)}
                    >
                        <CardContent sx={{ textAlign: 'center', p: 4 }}>
                            <QrCodeIcon sx={{ fontSize: 64, color: 'info.main', mb: 2 }} />
                            <Typography variant="h6" gutterBottom>
                                Simple Setup
                            </Typography>
                            <Chip label="Secure" color="info" size="small" sx={{ mb: 2 }} />
                            <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                Scan device QR code for automatic credential setup
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                • Scan QR/Data Matrix code<br/>
                                • Create managed thing with credentials<br/>
                                • Device auto-activates when plugged in<br/>
                                • No discovery process needed
                            </Typography>
                        </CardContent>
                    </Card>
                </Grid>
            </Grid>

            <Box sx={{ textAlign: 'center', mt: 4 }}>
                <Button
                    variant="outlined"
                    startIcon={<ArrowBack />}
                    onClick={handleCancel}
                    size="large"
                >
                    Back to Protocol Selection
                </Button>
            </Box>
        </Container>
    );

    // Render based on current step
    if (currentStep === OnboardingStep.PROTOCOL_SELECTION) {
        return (
            <Box sx={{ minHeight: '100vh', py: 4, bgcolor: 'background.default' }}>
                {renderProtocolSelection()}
            </Box>
        );
    }

    if (currentStep === OnboardingStep.METHOD_SELECTION) {
        return (
            <Box sx={{ minHeight: '100vh', py: 4, bgcolor: 'background.default' }}>
                {renderMethodSelection()}
            </Box>
        );
    }

    // OnboardingStep.EXECUTION - show the appropriate onboarding component
    return (
        <Box sx={{ minHeight: '100vh', py: 4, bgcolor: 'background.default' }}>
            {selectedMethod === OnboardingMethod.SIMPLE_ADDITION && (
                <SimpleDeviceOnboarding
                    protocol={selectedProtocol!}
                    onComplete={handleOnboardingComplete}
                    onCancel={handleCancel}
                />
            )}
            {selectedMethod === OnboardingMethod.QR_CODE_SCANNER && (
                <QRCodeOnboarding
                    protocol={selectedProtocol!}
                    onComplete={handleOnboardingComplete}
                    onCancel={handleCancel}
                />
            )}
        </Box>
    );
};

export default SimpleOnboarding;
