import React, { useState, useEffect } from 'react';
import {
    Box,
    Card,
    CardContent,
    Typography,
    Button,
    Alert,
    CircularProgress,
    Chip
} from '@mui/material';
import { validateEnvironmentConfig, logEnvironmentInfo } from '../utils/envValidation';
import { deviceService } from '../services/deviceService';
import { HUB_CONFIG } from '../constants';

const EnvironmentTest: React.FC = () => {
    const [envStatus, setEnvStatus] = useState<'loading' | 'valid' | 'invalid'>('loading');
    const [envError, setEnvError] = useState<string | null>(null);
    const [serviceTest, setServiceTest] = useState<'idle' | 'testing' | 'success' | 'error'>('idle');
    const [serviceError, setServiceError] = useState<string | null>(null);
    const [config, setConfig] = useState<any>(null);

    useEffect(() => {
        testEnvironment();
    }, []);

    const testEnvironment = () => {
        try {
            const envConfig = validateEnvironmentConfig();
            setConfig(envConfig);
            setEnvStatus('valid');
            setEnvError(null);
            logEnvironmentInfo();
        } catch (error) {
            setEnvStatus('invalid');
            setEnvError(error instanceof Error ? error.message : 'Unknown error');
        }
    };

    const testDeviceService = async () => {
        setServiceTest('testing');
        setServiceError(null);

        try {
            // Test with the configured hub ID
            console.log('Testing device service with hub ID:', HUB_CONFIG.MANAGED_THING_ID);
            
            const devices = await deviceService.listManagedThings(HUB_CONFIG.MANAGED_THING_ID);
            console.log('Device service test successful, found devices:', devices);
            
            setServiceTest('success');
        } catch (error) {
            console.error('Device service test failed:', error);
            setServiceTest('error');
            setServiceError(error instanceof Error ? error.message : 'Unknown error');
        }
    };

    return (
        <Box sx={{ maxWidth: 800, margin: '0 auto', p: 3 }}>
            <Typography variant="h4" gutterBottom>
                Environment Configuration Test
            </Typography>

            {/* Environment Variables Test */}
            <Card sx={{ mb: 3 }}>
                <CardContent>
                    <Typography variant="h6" gutterBottom>
                        Environment Variables
                    </Typography>
                    
                    {envStatus === 'loading' && (
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                            <CircularProgress size={20} />
                            <Typography>Validating environment...</Typography>
                        </Box>
                    )}

                    {envStatus === 'valid' && (
                        <Box>
                            <Alert severity="success" sx={{ mb: 2 }}>
                                Environment configuration is valid!
                            </Alert>
                            {config && (
                                <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <Typography variant="body2"><strong>Region:</strong></Typography>
                                        <Chip label={config.region} size="small" />
                                    </Box>
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <Typography variant="body2"><strong>Endpoint:</strong></Typography>
                                        <Typography variant="body2" sx={{ fontFamily: 'monospace', fontSize: '0.8em' }}>
                                            {config.endpoint}
                                        </Typography>
                                    </Box>
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <Typography variant="body2"><strong>Access Key:</strong></Typography>
                                        <Typography variant="body2" sx={{ fontFamily: 'monospace', fontSize: '0.8em' }}>
                                            {config.accessKeyId.substring(0, 8)}...
                                        </Typography>
                                    </Box>
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <Typography variant="body2"><strong>Session Token:</strong></Typography>
                                        <Typography variant="body2" sx={{ fontFamily: 'monospace', fontSize: '0.8em' }}>
                                            {config.sessionToken.substring(0, 20)}...
                                        </Typography>
                                    </Box>
                                </Box>
                            )}
                        </Box>
                    )}

                    {envStatus === 'invalid' && (
                        <Alert severity="error">
                            <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                                Environment Configuration Error
                            </Typography>
                            <Typography variant="body2">
                                {envError}
                            </Typography>
                        </Alert>
                    )}

                    <Button
                        variant="outlined"
                        onClick={testEnvironment}
                        sx={{ mt: 2 }}
                    >
                        Retest Environment
                    </Button>
                </CardContent>
            </Card>

            {/* Device Service Test */}
            <Card>
                <CardContent>
                    <Typography variant="h6" gutterBottom>
                        Device Service Connection Test
                    </Typography>

                    {serviceTest === 'idle' && (
                        <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                            Test the connection to AWS IoT Managed Integrations service
                        </Typography>
                    )}

                    {serviceTest === 'testing' && (
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, mb: 2 }}>
                            <CircularProgress size={20} />
                            <Typography>Testing device service connection...</Typography>
                        </Box>
                    )}

                    {serviceTest === 'success' && (
                        <Alert severity="success" sx={{ mb: 2 }}>
                            Device service connection successful! The AWS SDK is properly configured.
                        </Alert>
                    )}

                    {serviceTest === 'error' && (
                        <Alert severity="error" sx={{ mb: 2 }}>
                            <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                                Device Service Connection Error
                            </Typography>
                            <Typography variant="body2">
                                {serviceError}
                            </Typography>
                        </Alert>
                    )}

                    <Button
                        variant="contained"
                        onClick={testDeviceService}
                        disabled={envStatus !== 'valid' || serviceTest === 'testing'}
                    >
                        Test Device Service
                    </Button>
                </CardContent>
            </Card>

            <Box sx={{ mt: 3, textAlign: 'center' }}>
                <Typography variant="body2" color="text.secondary">
                    Check the browser console for detailed logs
                </Typography>
            </Box>
        </Box>
    );
};

export default EnvironmentTest;
