// src/pages/DeviceDetail.tsx

import React from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useSelector, useDispatch } from 'react-redux';
import {
    Card,
    CardContent,
    Typography,
    Button,
    Grid,
    Box
} from '@mui/material';
import { RootState } from '../store/store';
import { deviceService } from '../services/deviceService';
import { removeDevice } from '../store/deviceSlice';
import { deviceStatePollingService } from '../services/deviceStatePollingService';
import { deviceMonitorService } from '../services/deviceMonitorService';

// Mock device data
const mockDevice = {
    ManagedThingId: 'mock-123',
    Name: 'Mock Device',
    Protocol: 'ZIGBEE',
    currentState: {
        endpoints: [
            {
                endpointId: 'ep1',
                capabilities: [
                    {
                        id: 'cap1',
                        name: 'PowerController',
                        properties: [
                            { name: 'OnOff', value: true }
                        ]
                    }
                ]
            }
        ]
    }
};

const DeviceDetail: React.FC = () => {
    const { id } = useParams<{ id: string }>();
    const navigate = useNavigate();
    const dispatch = useDispatch();

    // Get device from Redux store by ManagedThingId
    const devices = useSelector((state: RootState) => state.devices.devices);

    // Set polling mode to device-detail when component mounts
    React.useEffect(() => {
        console.log(`DeviceDetail useEffect triggered with id: ${id}`);
        console.log(`Available devices in store:`, devices.map(d => ({ id: d.ManagedThingId, name: d.Name })));
        
        if (id) {
            console.log(`DeviceDetail mounted - setting polling mode to device-detail for ${id}`);
            
            const initializePolling = async () => {
                try {
                    console.log(`About to call setPollingMode for device: ${id}`);
                    await deviceStatePollingService.setPollingMode('device-detail', id);
                    console.log(`setPollingMode completed for device: ${id}`);
                    
                    deviceMonitorService.setMonitoringMode('device-detail');
                    console.log(`DeviceDetail polling initialized for ${id}`);
                    
                    // Check polling status
                    const status = deviceStatePollingService.getPollingStatus();
                    console.log(`Current polling status:`, status);
                } catch (error) {
                    console.error('Error initializing device detail polling:', error);
                }
            };
            
            initializePolling();
        } else {
            console.log('No device ID provided');
        }
        
        return () => {
            console.log('DeviceDetail unmounting - stopping polling');
            deviceStatePollingService.setPollingMode('stopped');
            deviceMonitorService.setMonitoringMode('stopped');
        };
    }, [id, devices]);
    const device = React.useMemo(() => {
        if (!id) return null;
        
        // Find device by ManagedThingId
        const foundDevice = devices.find(d => d.ManagedThingId === id);
        return foundDevice || null;
    }, [id, devices]);

    if (!device) {
        return (
            <Box sx={{ mt: 2 }}>
                <Card>
                    <CardContent>
                        <Typography>Device not found</Typography>
                        <Button onClick={() => navigate('/')} sx={{ mt: 2 }}>
                            Back to Dashboard
                        </Button>
                    </CardContent>
                </Card>
            </Box>
        );
    }

    const handleDelete = async () => {
        try {
            await deviceService.deleteDevice(device.ManagedThingId);
            dispatch(removeDevice(device.ManagedThingId));
            navigate('/');
        } catch (error) {
            console.error('Error deleting device:', error);
        }
    };

    // @ts-ignore
    return (
        <Box sx={{ mt: 2 }}>
            <Grid container spacing={3}>
                <Card>
                    <CardContent>
                        <Typography variant="h4" gutterBottom>
                            {device.Name}
                        </Typography>
                        <Typography color="textSecondary" gutterBottom>
                            ID: {device.ManagedThingId}
                        </Typography>
                        <Typography variant="h6" gutterBottom>
                            Protocol: {device.Protocol}
                        </Typography>
                        <Button
                            variant="contained"
                            color="error"
                            onClick={handleDelete}
                            sx={{ mt: 2 }}
                        >
                            Delete Device
                        </Button>
                    </CardContent>
                </Card>
            </Grid>
        </Box>
    );
};

export default DeviceDetail;
