// src/pages/Dashboard.tsx

import React, { useState, useEffect } from 'react';
import { 
    Grid, 
    Button, 
    Typography, 
    Box,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogActions,
    DialogContentText,
    Alert
} from '@mui/material';
import * as Icons from '@mui/icons-material';
import { Link } from 'react-router-dom';
import { useSelector, useDispatch } from 'react-redux';
import DeviceWidget from '../components/devices/DeviceWidget';
import DeviceInitialization from '../components/devices/DeviceInitialization';
import { RootState } from '../store/store';
import { deviceService } from '../services/deviceService';
import { resetDevices } from '../store/deviceSlice';
import { deviceStatePollingService } from '../services/deviceStatePollingService';
import { ProtocolType } from '../types/deviceTypes';

const Dashboard: React.FC = () => {
    const devices = useSelector((state: RootState) => state.devices.devices);
    const [showInitialization, setShowInitialization] = useState(true);
    const [showResetConfirmation, setShowResetConfirmation] = useState(false);
    const hubId = useSelector((state: RootState) => state.hub.hubId);
    const dispatch = useDispatch();

    // Only show initialization on first load, not when returning from onboarding
    useEffect(() => {
        // Only show initialization if there are truly no devices and we haven't just completed onboarding
        if (devices.length === 0) {
            setShowInitialization(true);
        } else {
            setShowInitialization(false);
        }
    }, [devices.length]);

    // Device states are now managed by the polling service - no need to manually load them
    // The deviceStatePollingService handles state updates via polling

    const handleInitializationComplete = () => {
        setShowInitialization(false);
    };

    const handleReset = async () => {
        try {
            console.log('Starting reset all devices process...');
            
            // Delete all devices from AWS first
            const deletePromises = devices.map(async (device) => {
                try {
                    console.log(`Deleting device ${device.ManagedThingId} (${device.Name}) from AWS...`);
                    await deviceService.deleteDevice(device.ManagedThingId);
                    console.log(`Successfully deleted device ${device.ManagedThingId} from AWS`);
                } catch (error) {
                    console.error(`Failed to delete device ${device.ManagedThingId} from AWS:`, error);
                    // Continue with other devices even if one fails
                }
            });
            
            // Wait for all deletions to complete
            await Promise.allSettled(deletePromises);
            console.log('All device deletion attempts completed');
            
            // Clear Redux store after AWS deletions
            dispatch(resetDevices());
            
            // Stop polling service to prevent it from trying to poll deleted devices
            deviceStatePollingService.setPollingMode('stopped');
            
            // Show initialization screen
            setShowInitialization(true);
            setShowResetConfirmation(false);
            
            console.log('Reset all devices completed successfully');
        } catch (error) {
            console.error('Error during reset all devices:', error);
            // Even if there are errors, still clear the Redux store and show initialization
            dispatch(resetDevices());
            setShowInitialization(true);
            setShowResetConfirmation(false);
        }
    };

    // Set polling mode to dashboard when component mounts
    useEffect(() => {
        const startDashboardPolling = async () => {
            console.log('Dashboard mounted - starting dashboard polling (ListManagedThings every 5min, ReadState+GetState every 1min)');
            console.log('Current hubId:', hubId);
            
            if (!hubId) {
                console.error('No hubId available for dashboard polling');
                return;
            }
            
            try {
                await deviceStatePollingService.setPollingMode('dashboard');
                console.log('Dashboard polling mode set successfully');
                
                // Check polling status
                const status = deviceStatePollingService.getPollingStatus();
                console.log('Polling status after setting dashboard mode:', status);
            } catch (error) {
                console.error('Error setting dashboard polling mode:', error);
            }
        };
        
        startDashboardPolling();
        
        return () => {
            console.log('Dashboard unmounting - stopping polling');
            deviceStatePollingService.setPollingMode('stopped');
        };
    }, [hubId]);

    return (
        <Box sx={{ mt: 2 }}>
            {showInitialization && hubId && (
                <DeviceInitialization 
                    hubId={hubId} 
                    onComplete={handleInitializationComplete}
                />
            )}

            <Box sx={{ mb: 4, display: 'flex', alignItems: 'center' }}>
                <Typography variant="h4" sx={{ flexGrow: 1 }}>
                    My Devices
                </Typography>
                <Button
                    variant="contained"
                    color="primary"
                    component={Link}
                    to="/simple-onboard"
                    sx={{ mr: 2 }}
                >
                    Add Device
                </Button>
                <Button
                    variant="outlined"
                    color="error"
                    onClick={() => setShowResetConfirmation(true)}
                >
                    Delete All
                </Button>
            </Box>

            <Dialog
                open={showResetConfirmation}
                onClose={() => setShowResetConfirmation(false)}
                maxWidth="sm"
                fullWidth
            >
                <DialogTitle>
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                        <Icons.Warning color="warning" />
                        Confirm Delete All Devices
                    </Box>
                </DialogTitle>
                <DialogContent>
                    <DialogContentText sx={{ mb: 2 }}>
                        <strong>Warning:</strong> This will permanently delete all {devices.length} device{devices.length !== 1 ? 's' : ''} from AWS and remove them from your networks.
                    </DialogContentText>
                    
                    {/* Check if there are any Z-Wave devices */}
                    {devices.some(device => device.Protocol === ProtocolType.ZWAVE) && (
                        <>
                            <Alert severity="warning" sx={{ mb: 2 }}>
                                <strong>Z-Wave devices detected!</strong> You have Z-Wave devices that require manual intervention.
                            </Alert>
                            
                            <DialogContentText sx={{ mb: 2 }}>
                                <strong>For Z-Wave devices, you must:</strong>
                            </DialogContentText>
                            <Box component="ol" sx={{ pl: 2, mb: 2 }}>
                                <Box component="li" sx={{ mb: 1 }}>
                                    Click "Delete All Devices" to start the deletion process
                                </Box>
                                <Box component="li" sx={{ mb: 1 }}>
                                    Physically access each Z-Wave device
                                </Box>
                                <Box component="li" sx={{ mb: 1 }}>
                                    Put each device into pairing/exclusion mode (refer to device manuals)
                                </Box>
                                <Box component="li">
                                    Devices will be fully removed from the Z-Wave network
                                </Box>
                            </Box>
                        </>
                    )}
                    
                    <DialogContentText>
                        You will need to re-pair all devices after this operation. This action cannot be undone.
                    </DialogContentText>
                </DialogContent>
                <DialogActions>
                    <Button onClick={() => setShowResetConfirmation(false)} color="primary">
                        Cancel
                    </Button>
                    <Button 
                        onClick={handleReset} 
                        color="error" 
                        variant="contained"
                        startIcon={<Icons.Delete />}
                    >
                        Delete All Devices
                    </Button>
                </DialogActions>
            </Dialog>

            <Grid container spacing={3}>
                {devices.length === 0 ? (
                    <Box sx={{ width: '100%', p: 2 }}>
                        <Typography>No devices found. Add a device to get started.</Typography>
                    </Box>
                ) : (
                    devices.map((device) => (
                        <Grid
                            key={device.ManagedThingId}
                            sx={{
                                width: {
                                    xs: '100%',
                                    sm: '50%',
                                    md: '33.33%'
                                },
                                p: 1
                            }}
                        >
                            <DeviceWidget device={device} />
                        </Grid>
                    ))
                )}
            </Grid>
        </Box>
    );
};

export default Dashboard;
