import React, { useState } from 'react';
import {
    TextField,
    Box,
    Typography,
    Alert,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogActions,
    Button,
    InputAdornment,
    IconButton
} from '@mui/material';
import WarningIcon from '@mui/icons-material/Warning';
import QrCodeScannerIcon from '@mui/icons-material/QrCodeScanner';
import {ProtocolType} from "../../../types/deviceTypes";
import {AuthMethod, AuthDetails} from "../../../types/authTypes";
import SimpleQRScanner from '../../SimpleQRScanner';
import { deviceService } from '../../../services/deviceService';
import { HUB_CONFIG } from '../../../constants';


interface Props {
    protocol: ProtocolType;
    authMethod: AuthMethod;
    authDetails?: AuthDetails;
    onSubmit: (authDetails: AuthDetails) => void;
    onNext: (thingId: string) => void;
}

const InclusionWarningDialog: React.FC<{
    open: boolean;
    protocol: ProtocolType;
    onConfirm: () => void;
    onCancel: () => void;
}> = ({ open, protocol, onConfirm, onCancel }) => (
    <Dialog
        open={open}
        onClose={onCancel}
        maxWidth="sm"
        fullWidth
    >
        <DialogTitle sx={{
            display: 'flex',
            alignItems: 'center',
            gap: 1,
            color: 'warning.main'
        }}>
            <WarningIcon />
            {protocol} Inclusion Mode Warning
        </DialogTitle>
        <DialogContent>
            <Alert severity="warning" sx={{ mb: 2 }}>
                Using Inclusion Mode is potentially unsafe!
            </Alert>
            <Typography paragraph>
                By proceeding with {protocol} Inclusion Mode:
            </Typography>
            <Box component="ul" sx={{ mt: 1 }}>
                <Typography component="li">
                    Any {protocol} device in pairing mode within range can connect
                </Typography>
                <Typography component="li">
                    This could allow unauthorized or malicious devices to join your network
                </Typography>
                <Typography component="li">
                    It's recommended to use this only in controlled environments
                </Typography>
            </Box>
        </DialogContent>
        <DialogActions>
            <Button onClick={onCancel} variant="outlined">
                Cancel
            </Button>
            <Button
                onClick={onConfirm}
                variant="contained"
                color="warning"
                autoFocus
            >
                I Understand, Proceed Anyway
            </Button>
        </DialogActions>
    </Dialog>
);

export const DeviceAuthentication: React.FC<Props> = ({
                                                          protocol,
                                                          authMethod,
                                                          authDetails,
                                                          onSubmit,
                                                          onNext
                                                      }) => {
    const [AuthDetails, setAuthDetails] = useState<AuthDetails>(authDetails || {});
    const [showWarning, setShowWarning] = useState(false);
    const [pendingData, setPendingData] = useState<AuthDetails | null>(null);
    const [showQRScanner, setShowQRScanner] = useState(false);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [isProcessing, setIsProcessing] = useState(false);

    const handleChange = (field: keyof AuthDetails) => (
        event: React.ChangeEvent<HTMLInputElement>
    ) => {
        const newData = {
            ...AuthDetails,
            [field]: event.target.value
        };
        setAuthDetails(newData);

        // Only submit immediately if not in inclusion mode
        if (!isInclusionMode(authMethod)) {
            onSubmit(newData);
        }
    };

    const handleQRScan = (result: string) => {
        const newData = {
            ...AuthDetails,
            dsk: result
        };
        setAuthDetails(newData);
        onSubmit(newData);
    };

    const isInclusionMode = (method: AuthMethod): boolean => {
        return method === AuthMethod.ZWAVE_INCLUSION ||
            method === AuthMethod.ZIGBEE_INCLUSION;
    };

    const handleSubmit = (data: AuthDetails) => {
        if (isInclusionMode(authMethod)) {
            setPendingData(data);
            setShowWarning(true);
        } else {
            onSubmit(data);
        }
    };

    const handleWarningConfirm = async () => {
        setShowWarning(false);
        setPendingData(null);
        
        // Start the simple inclusion process
        await handleSimpleInclusion();
    };

    /**
     * Handle simple inclusion for Z-Wave and Zigbee devices without authentication material
     */
    const handleSimpleInclusion = async () => {
        try {
            setIsProcessing(true);
            setError(null);

            const protocolType = authMethod === AuthMethod.ZWAVE_INCLUSION ? 'ZWAVE' : 'ZIGBEE';
            
            console.log(`Starting ${protocolType} simple inclusion...`);
            
            // Start device discovery (this puts the hub in discovery mode)
            const result = await deviceService.startSimpleDeviceDiscovery(protocolType, HUB_CONFIG.MANAGED_THING_ID);
            
            console.log(`${protocol} device discovery started with ID:`, result.discoveryId);
            
            // For now, we'll simulate the discovery process
            // In a real implementation, you'd poll for discovered devices and let user select
            setTimeout(async () => {
                try {
                    // List discovered devices
                    const discoveredDevices = await deviceService.listDiscoveredDevices(result.discoveryId);
                    
                    if (discoveredDevices.length > 0) {
                        // For simplicity, auto-select the first discovered device
                        const firstDevice = discoveredDevices[0];
                        
                        // Create managed thing from discovered device
                        const managedThingResult = await deviceService.createManagedThingFromDiscoveredDevice({
                            discoveryId: result.discoveryId,
                            deviceIdentifier: firstDevice.DeviceIdentifier || firstDevice.NodeId,
                            protocol: protocolType,
                            authenticationMaterial: firstDevice.AuthenticationMaterial
                        });
                        
                        // Proceed to next step with the managed thing ID
                        onNext(managedThingResult.managedThingId);
                    } else {
                        setError('No devices were discovered. Please ensure your device is in pairing mode.');
                    }
                } catch (err) {
                    console.error('Error during device discovery:', err);
                    setError(err instanceof Error ? err.message : 'Failed to discover devices');
                } finally {
                    setIsProcessing(false);
                }
            }, 5000); // Wait 5 seconds for discovery
            
        } catch (err) {
            console.error(`Error during ${protocol} simple inclusion:`, err);
            setError(err instanceof Error ? err.message : `Failed to start ${protocol} discovery`);
            setIsProcessing(false);
        }
    };

    const handleDSKSubmit = async () => {
        if (!AuthDetails.dsk) return;

        try {
            setLoading(true);
            setError(null);

            // Create Z-Wave device with DSK
            const device = await deviceService.createZWaveDevice(AuthDetails.dsk);

            // Send initial ReadState to populate endpoints
            try {
                console.log('Sending initial ReadState after Z-Wave device creation...');
                await deviceService.sendInitialReadStateAfterOnboarding(device.ManagedThingId);
                console.log('Initial ReadState sent for Z-Wave device');
            } catch (error) {
                console.warn('Failed to send initial ReadState for Z-Wave device (non-critical):', error);
            }

            // If we get here, the device was created successfully
            // We can proceed with the next step
            onNext(device.ManagedThingId);
        } catch (err) {
            console.error('Error creating Z-Wave device:', err);
            setError(err instanceof Error ? err.message : 'Failed to create Z-Wave device');
        } finally {
            setLoading(false);
        }
    };

    const handleInstallCodeSubmit = async () => {
        if (!AuthDetails.installCode) return;

        try {
            setLoading(true);
            setError(null);

            // Create Zigbee device with install code
            const result = await deviceService.createZigbeeDeviceWithInstallCode(AuthDetails.installCode);

            // Send initial ReadState to populate endpoints
            try {
                console.log('Sending initial ReadState after Zigbee device creation...');
                await deviceService.sendInitialReadStateAfterOnboarding(result.managedThingId);
                console.log('Initial ReadState sent for Zigbee device');
            } catch (error) {
                console.warn('Failed to send initial ReadState for Zigbee device (non-critical):', error);
            }

            // Proceed to next step
            onNext(result.managedThingId);
        } catch (err) {
            console.error('Error creating Zigbee device with install code:', err);
            setError(err instanceof Error ? err.message : 'Failed to create Zigbee device');
        } finally {
            setLoading(false);
        }
    };

    const renderAuthFields = () => {
        switch (authMethod) {
            case AuthMethod.ZWAVE_PIN:
                AuthDetails.mode= 'SECURE'
                return (
                    <TextField
                        fullWidth
                        label="PIN Code"
                        value={AuthDetails.pinCode || ''}
                        onChange={handleChange('pinCode')}
                        margin="normal"
                        helperText="Enter the Z-Wave device PIN code"
                    />
                );
            case AuthMethod.ZWAVE_DSK:
                AuthDetails.mode= 'SECURE'
                return (
                    <Box>
                        <TextField
                            fullWidth
                            label="DSK"
                            value={AuthDetails.dsk || ''}
                            onChange={handleChange('dsk')}
                            margin="normal"
                            helperText="Enter the Z-Wave device DSK or scan QR code"
                            InputProps={{
                                endAdornment: (
                                    <InputAdornment position="end">
                                        <IconButton
                                            onClick={() => setShowQRScanner(true)}
                                            edge="end"
                                        >
                                            <QrCodeScannerIcon />
                                        </IconButton>
                                    </InputAdornment>
                                ),
                            }}
                        />
                        {showQRScanner && (
                            <Box sx={{ mt: 2, mb: 2 }}>
                                <SimpleQRScanner
                                    onScanSuccess={(decodedText: string) => {
                                        handleQRScan(decodedText);
                                        setShowQRScanner(false);
                                    }}
                                    width={400}
                                    height={300}
                                />
                                <Button
                                    variant="outlined"
                                    onClick={() => setShowQRScanner(false)}
                                    sx={{ mt: 1 }}
                                >
                                    Close Scanner
                                </Button>
                            </Box>
                        )}
                        <Button
                            variant="contained"
                            onClick={handleDSKSubmit}
                            disabled={!AuthDetails.dsk || loading}
                            sx={{ mt: 2 }}
                        >
                            {loading ? 'Creating Device...' : 'Create Z-Wave Device'}
                        </Button>
                    </Box>
                );
            case AuthMethod.ZIGBEE_INSTALL_CODE:
                AuthDetails.mode= 'SECURE'
                return (
                    <Box>
                        <TextField
                            fullWidth
                            label="Install Code"
                            value={AuthDetails.installCode || ''}
                            onChange={handleChange('installCode')}
                            margin="normal"
                            helperText="Enter the Zigbee device install code"
                        />
                        <Button
                            variant="contained"
                            onClick={handleInstallCodeSubmit}
                            disabled={!AuthDetails.installCode || loading}
                            sx={{ mt: 2 }}
                        >
                            {loading ? 'Creating Device...' : 'Create Zigbee Device'}
                        </Button>
                    </Box>
                );
            case AuthMethod.ZWAVE_INCLUSION:
            case AuthMethod.ZIGBEE_INCLUSION:
                AuthDetails.mode= 'UNSECURE'
                return (
                    <Box>
                        <Alert
                            severity="warning"
                            icon={<WarningIcon />}
                            sx={{ mt: 2, mb: 2 }}
                        >
                            <Typography variant="subtitle1" sx={{ fontWeight: 'bold', mb: 1 }}>
                                Warning: Unsecure {protocol} Inclusion Mode
                            </Typography>
                            <Typography paragraph>
                                Using unsecure inclusion mode has the following risks:
                            </Typography>
                            <Box component="ul" sx={{ mt: 1, mb: 1 }}>
                                <Typography component="li">
                                    Any {protocol} device in pairing mode within range can connect
                                </Typography>
                                <Typography component="li">
                                    Unauthorized or malicious devices could join your network
                                </Typography>
                                <Typography component="li">
                                    Only use this in controlled environments
                                </Typography>
                            </Box>
                        </Alert>

                        <Alert severity="info" sx={{ mb: 2 }}>
                            <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                                Simple Inclusion Process:
                            </Typography>
                            <Typography paragraph>
                                1. Click "Start {protocol} Inclusion" below
                            </Typography>
                            <Typography paragraph>
                                2. Put your {protocol} device in pairing/inclusion mode
                            </Typography>
                            <Typography paragraph>
                                3. The hub will automatically discover and add the device
                            </Typography>
                        </Alert>

                        <Button
                            variant="contained"
                            color="warning"
                            onClick={() => {
                                setPendingData({});
                                setShowWarning(true);
                            }}
                            disabled={isProcessing}
                            sx={{ mt: 2 }}
                        >
                            {isProcessing ? `Starting ${protocol} Inclusion...` : `Start ${protocol} Inclusion`}
                        </Button>

                        {error && (
                            <Alert severity="error" sx={{ mt: 2 }}>
                                {error}
                            </Alert>
                        )}
                    </Box>
                );
            default:
                return null;
        }
    };

    return (
        <Box>
            {renderAuthFields()}

            <InclusionWarningDialog
                open={showWarning}
                protocol={protocol}
                onConfirm={handleWarningConfirm}
                onCancel={() => {
                    setShowWarning(false);
                    setPendingData(null);
                }}
            />
        </Box>
    );
};
