import React, { useState, useEffect } from 'react';
import {
    Box,
    Card,
    CardContent,
    Typography,
    Button,
    Alert,
    Stepper,
    Step,
    StepLabel,
    CircularProgress,
    Chip,
    Divider,
    TextField
} from '@mui/material';
import {
    Wifi,
    Security,
    CheckCircle,
    Error as ErrorIcon,
    DeviceHub
} from '@mui/icons-material';
import { ProtocolType } from '../../types/deviceTypes';
import { AuthMethod } from '../../types/authTypes';
import { deviceService } from '../../services/deviceService';
import { HUB_CONFIG } from '../../constants';

interface SimpleDeviceOnboardingProps {
    protocol: ProtocolType;
    onComplete: (managedThingId: string, deviceName?: string) => void;
    onCancel: () => void;
}

enum OnboardingStep {
    SELECT_METHOD = 0,
    ZWAVE_PIN_SETUP = 1,
    DEVICE_CREATION = 2,
    DISCOVERY = 3,
    NAMING = 4,
    COMPLETE = 5
}

interface OnboardingState {
    step: OnboardingStep;
    authMethod: AuthMethod | null;
    zwavePinCode: string | null;
    zwaveSecureMode: boolean;
    discoveryId: string | null;
    discoveredDevices: any[];
    selectedDevice: any | null;
    managedThingId: string | null;
    deviceName: string;
    loading: boolean;
    error: string | null;
    discoveryStarted: boolean;
    pollingInterval: NodeJS.Timeout | null;
}

const SimpleDeviceOnboarding: React.FC<SimpleDeviceOnboardingProps> = ({
    protocol,
    onComplete,
    onCancel
}) => {
    const [state, setState] = useState<OnboardingState>({
        step: protocol === ProtocolType.ZWAVE ? OnboardingStep.ZWAVE_PIN_SETUP : OnboardingStep.DEVICE_CREATION,
        authMethod: protocol === ProtocolType.ZWAVE ? AuthMethod.ZWAVE_INCLUSION : AuthMethod.ZIGBEE_INCLUSION,
        zwavePinCode: null,
        zwaveSecureMode: true,
        discoveryId: null,
        discoveredDevices: [],
        selectedDevice: null,
        managedThingId: null,
        deviceName: '',
        loading: false,
        error: null,
        discoveryStarted: false,
        pollingInterval: null
    });

    // Cleanup polling interval on unmount
    useEffect(() => {
        return () => {
            if (state.pollingInterval) {
                clearInterval(state.pollingInterval);
            }
        };
    }, [state.pollingInterval]);

    const steps = protocol === ProtocolType.ZWAVE ? [
        'Enter PIN Code',
        'Device Discovery',
        'Name Device',
        'Complete'
    ] : [
        'Create Device',
        'Device Discovery',
        'Name Device',
        'Complete'
    ];

    // Map internal OnboardingStep enum to correct stepper index based on protocol
    const getStepperIndex = (step: OnboardingStep, protocol: ProtocolType): number => {
        if (protocol === ProtocolType.ZWAVE) {
            switch (step) {
                case OnboardingStep.ZWAVE_PIN_SETUP: return 0; // "Enter PIN Code"
                case OnboardingStep.DISCOVERY: return 1;        // "Device Discovery"
                case OnboardingStep.NAMING: return 2;           // "Name Device"
                case OnboardingStep.COMPLETE: return 3;         // "Complete"
                default: return 0;
            }
        } else { // ZIGBEE
            switch (step) {
                case OnboardingStep.DEVICE_CREATION: return 0;  // "Create Device"
                case OnboardingStep.DISCOVERY: return 1;        // "Device Discovery"
                case OnboardingStep.NAMING: return 2;           // "Name Device"
                case OnboardingStep.COMPLETE: return 3;         // "Complete"
                default: return 0;
            }
        }
    };


    const handleSimpleInclusion = async () => {
        try {
            setState(prev => ({ ...prev, loading: true, error: null }));

            // Start device discovery (this puts the hub in discovery mode)
            console.log(`Starting ${protocol} device discovery...`);
            const protocolType = state.authMethod === AuthMethod.ZWAVE_INCLUSION ? 'ZWAVE' : 'ZIGBEE';
            const pinCode = protocolType === 'ZWAVE' ? (state.zwavePinCode || '00000') : undefined;
            const result = await deviceService.startSimpleDeviceDiscovery(protocolType, HUB_CONFIG.MANAGED_THING_ID, pinCode);
            
            setState(prev => ({
                ...prev,
                discoveryId: result.discoveryId,
                step: OnboardingStep.DISCOVERY,
                loading: false,
                discoveryStarted: true
            }));

            // Start polling for discovered devices
            startPollingForDevices(result.discoveryId);

        } catch (error) {
            console.error('Error during simple inclusion:', error);
            setState(prev => ({
                ...prev,
                loading: false,
                error: error instanceof Error ? error.message : 'Failed to start device discovery'
            }));
        }
    };

    const startPollingForDevices = (discoveryId: string) => {
        const interval = setInterval(async () => {
            try {
                const discoveredDevices = await deviceService.listDiscoveredDevices(discoveryId);
                console.log('Polling result - discoveredDevices:', discoveredDevices);
                console.log('Polling result - discoveredDevices.length:', discoveredDevices.length);
                
                setState(prev => ({
                    ...prev,
                    discoveredDevices
                }));

                // If we found devices, automatically create managed thing
                if (discoveredDevices.length > 0) {
                    console.log('Found discovered devices, starting automatic creation:', discoveredDevices);
                    
                    // Stop polling
                    clearInterval(interval);
                    
                    // Automatically create managed thing from first discovered device
                    const firstDevice = discoveredDevices[0];
                    console.log('First device to create managed thing from:', firstDevice);
                    
                    const protocolType = state.authMethod === AuthMethod.ZWAVE_INCLUSION ? 'ZWAVE' : 'ZIGBEE';
                    console.log('Protocol type:', protocolType);
                    
                    setState(prev => ({ ...prev, loading: true, selectedDevice: firstDevice }));
                    
                    try {
                        const result = await deviceService.createManagedThingFromDiscoveredDevice({
                            discoveryId: discoveryId,
                            deviceIdentifier: firstDevice.DeviceIdentifier || firstDevice.NodeId,
                            protocol: protocolType,
                            authenticationMaterial: firstDevice.AuthenticationMaterial
                        });

                        console.log('Managed thing created automatically:', result.managedThingId);
                        
                        // Send initial ReadState to populate endpoints (with delay for activation)
                        setTimeout(async () => {
                            try {
                                console.log('Sending initial ReadState after 2-second delay for device activation...');
                                await deviceService.sendInitialReadStateAfterOnboarding(result.managedThingId);
                                console.log('Initial ReadState sent successfully');
                            } catch (error) {
                                console.warn('Failed to send initial ReadState (non-critical):', error);
                            }
                        }, 2000); // 2-second delay to allow device activation

                        // Update state and go to naming step
                        setState(prev => ({
                            ...prev,
                            managedThingId: result.managedThingId,
                            step: OnboardingStep.NAMING,
                            loading: false,
                            pollingInterval: null
                        }));
                    } catch (error) {
                        console.error('Error creating managed thing:', error);
                        setState(prev => ({
                            ...prev,
                            error: `Failed to create managed thing: ${error instanceof Error ? error.message : 'Unknown error'}`,
                            loading: false
                        }));
                    }
                }
            } catch (error) {
                console.error('Error polling for discovered devices:', error);
            }
        }, 2000); // Poll every 2 seconds

        setState(prev => ({
            ...prev,
            pollingInterval: interval
        }));

        // Stop polling after 180 seconds
        setTimeout(() => {
            clearInterval(interval);
            setState(prev => ({
                ...prev,
                pollingInterval: null,
                error: 'Discovery timed out after 3 minutes. No devices were found. Please ensure your device is in pairing mode and try again.'
            }));
            console.log('Discovery polling timed out after 180 seconds');
        }, 180000);
    };

    const handleDeviceSelect = async (device: any) => {
        try {
            setState(prev => ({ ...prev, loading: true, selectedDevice: device }));

            if (!state.discoveryId) {
                throw new Error('No discovery ID available');
            }

            // Create managed thing from discovered device
            const protocolType = state.authMethod === AuthMethod.ZWAVE_INCLUSION ? 'ZWAVE' : 'ZIGBEE';
            const result = await deviceService.createManagedThingFromDiscoveredDevice({
                discoveryId: state.discoveryId,
                deviceIdentifier: device.DeviceIdentifier || device.NodeId,
                protocol: protocolType,
                authenticationMaterial: device.AuthenticationMaterial
            });

            // Clear polling interval
            if (state.pollingInterval) {
                clearInterval(state.pollingInterval);
            }

            setState(prev => ({
                ...prev,
                managedThingId: result.managedThingId,
                step: OnboardingStep.COMPLETE,
                loading: false,
                pollingInterval: null
            }));

        } catch (error) {
            console.error('Error creating managed thing from discovered device:', error);
            setState(prev => ({
                ...prev,
                loading: false,
                error: error instanceof Error ? error.message : 'Failed to create device'
            }));
        }
    };

    const handleComplete = () => {
        if (state.managedThingId) {
            onComplete(state.managedThingId, state.deviceName.trim() || undefined);
        }
    };

    const handleNameSubmit = async () => {
        if (!state.deviceName.trim()) {
            setState(prev => ({ ...prev, error: 'Please enter a device name' }));
            return;
        }

        try {
            setState(prev => ({ ...prev, loading: true, error: null }));

            // Update the device name in Redux store
            if (state.managedThingId) {
                // Import dispatch from Redux
                const { useDispatch } = require('react-redux');
                const { updateDeviceName } = require('../../store/deviceSlice');
                
                // We'll handle this differently - just proceed to completion
                // The name will be handled when the device is added to the store
                console.log(`Device named: ${state.deviceName}`);
                
                setState(prev => ({
                    ...prev,
                    step: OnboardingStep.COMPLETE,
                    loading: false
                }));
            }
        } catch (error) {
            console.error('Error saving device name:', error);
            setState(prev => ({
                ...prev,
                loading: false,
                error: error instanceof Error ? error.message : 'Failed to save device name'
            }));
        }
    };


    const renderDeviceCreation = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Create {protocol} Device
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Creating a managed device entry for simple inclusion.
            </Typography>

            <Alert severity="warning" sx={{ mb: 3 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                    Security Warning
                </Typography>
                <Typography variant="body2">
                    Simple inclusion mode allows any {protocol} device in pairing mode to connect. 
                    Only use this in controlled environments.
                </Typography>
            </Alert>

            <Button
                variant="contained"
                color="warning"
                onClick={handleSimpleInclusion}
                disabled={state.loading}
                startIcon={state.loading ? <CircularProgress size={20} /> : <DeviceHub />}
                fullWidth
                size="large"
            >
                {state.loading ? 'Starting Discovery...' : 'Start Discovery'}
            </Button>

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderDiscovery = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                {protocol} Device Discovery
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                The hub is now in {protocol === ProtocolType.ZWAVE ? 'secure ' : ''}discovery mode. Put your {protocol} device in pairing mode.
            </Typography>

            <Alert severity="info" sx={{ mb: 3 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                    Next Steps:
                </Typography>
                <Typography component="div">
                    1. Put your {protocol} device in pairing/inclusion mode<br/>
                    2. Wait for the device to be discovered (up to 3 minutes)<br/>
                    3. The device will be added automatically when found
                </Typography>
            </Alert>

            {state.discoveryId && (
                <Alert severity="success" sx={{ mb: 2 }}>
                    <Typography variant="body2">
                        <strong>Discovery ID:</strong> {state.discoveryId}
                        {protocol === ProtocolType.ZWAVE && (
                            <><br/><strong>PIN Code:</strong> {state.zwavePinCode || '00000'}</>
                        )}
                    </Typography>
                </Alert>
            )}

            {state.loading && (
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, mb: 2 }}>
                    <CircularProgress size={24} />
                    <Typography>
                        {state.selectedDevice ? 'Creating managed device...' : `Discovering ${protocol} devices...`}
                    </Typography>
                </Box>
            )}

            {state.discoveredDevices.length > 0 && (
                <Box sx={{ mb: 3 }}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 'bold' }}>
                        Discovered Devices ({state.discoveredDevices.length})
                    </Typography>
                    {state.discoveredDevices.map((device, index) => (
                        <Card 
                            key={device.DeviceIdentifier || device.NodeId || index}
                            sx={{ 
                                mb: 2,
                                border: '2px solid',
                                borderColor: 'success.main'
                            }}
                        >
                            <CardContent>
                                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                                    <Box>
                                        <Typography variant="subtitle1">
                                            {device.DeviceName || `${protocol} Device`}
                                        </Typography>
                                        <Typography variant="body2" color="text.secondary">
                                            ID: {device.DeviceIdentifier || device.NodeId}
                                        </Typography>
                                        {device.ManufacturerName && (
                                            <Typography variant="body2" color="text.secondary">
                                                Manufacturer: {device.ManufacturerName}
                                            </Typography>
                                        )}
                                        {device.ProductName && (
                                            <Typography variant="body2" color="text.secondary">
                                                Product: {device.ProductName}
                                            </Typography>
                                        )}
                                    </Box>
                                    <Chip 
                                        label={state.loading ? "Adding..." : "Found!"} 
                                        color="success" 
                                        icon={state.loading ? <CircularProgress size={16} /> : <CheckCircle />}
                                    />
                                </Box>
                            </CardContent>
                        </Card>
                    ))}
                </Box>
            )}

            {state.discoveryStarted && state.discoveredDevices.length === 0 && !state.loading && (
                <Alert severity="warning" sx={{ mb: 2 }}>
                    <Typography variant="body2">
                        No devices discovered yet. Make sure your {protocol} device is in pairing mode and within range.
                        {protocol === ProtocolType.ZWAVE && (
                            <> The PIN code being used is: <strong>{state.zwavePinCode || '00000'}</strong></>
                        )}
                    </Typography>
                </Alert>
            )}

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderComplete = () => (
        <Box sx={{ textAlign: 'center' }}>
            <CheckCircle color="success" sx={{ fontSize: 64, mb: 2 }} />
            <Typography variant="h6" gutterBottom>
                Device Onboarding Complete!
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Your {protocol} device has been successfully added to the network.
            </Typography>

            {state.managedThingId && (
                <Alert severity="success" sx={{ mb: 3 }}>
                    <Typography variant="body2">
                        <strong>Device ID:</strong> {state.managedThingId}
                    </Typography>
                </Alert>
            )}

            <Button
                variant="contained"
                onClick={handleComplete}
                size="large"
            >
                Continue to Device Management
            </Button>
        </Box>
    );

    const renderNaming = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Name Your Device
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Give your {protocol} device a memorable name to make it easier to identify.
            </Typography>

            {state.managedThingId && (
                <Alert severity="success" sx={{ mb: 3 }}>
                    <Typography variant="body2">
                        <strong>Device ID:</strong> {state.managedThingId}
                    </Typography>
                </Alert>
            )}

            <TextField
                fullWidth
                label="Device Name"
                value={state.deviceName}
                onChange={(e) => setState(prev => ({ ...prev, deviceName: e.target.value, error: null }))}
                placeholder={`My ${protocol} Device`}
                helperText="Enter a name that will help you identify this device"
                sx={{ mb: 3 }}
                onKeyPress={(e) => e.key === 'Enter' && handleNameSubmit()}
            />

            <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 3 }}>
                <Button
                    variant="outlined"
                    onClick={() => setState(prev => ({ ...prev, step: OnboardingStep.DISCOVERY }))}
                    disabled={state.loading}
                >
                    Back
                </Button>
                <Button
                    variant="contained"
                    onClick={handleNameSubmit}
                    disabled={state.loading || !state.deviceName.trim()}
                    startIcon={state.loading ? <CircularProgress size={20} /> : undefined}
                >
                    {state.loading ? 'Saving...' : 'Save Name'}
                </Button>
            </Box>

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderZWavePinSetup = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Choose Z-Wave Connection Type
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Select how you want to connect your Z-Wave device to the network.
            </Typography>

            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2, mb: 3 }}>
                <Card 
                    sx={{ 
                        cursor: 'pointer', 
                        border: state.zwavePinCode !== null ? '2px solid' : '1px solid',
                        borderColor: state.zwavePinCode !== null ? 'success.main' : 'divider',
                        '&:hover': { bgcolor: 'action.hover' } 
                    }}
                    onClick={() => setState(prev => ({ ...prev, zwavePinCode: prev.zwavePinCode || '' }))}
                >
                    <CardContent>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                            <Security color={state.zwavePinCode !== null ? 'success' : 'disabled'} />
                            <Box sx={{ flex: 1 }}>
                                <Typography variant="subtitle1" sx={{ fontWeight: 'bold' }}>
                                    Secure Inclusion (Recommended)
                                </Typography>
                                <Typography variant="body2" color="text.secondary">
                                    Uses your device's PIN code for encrypted communication
                                </Typography>
                                <Chip label="Secure" color="success" size="small" sx={{ mt: 1 }} />
                            </Box>
                        </Box>
                    </CardContent>
                </Card>

                <Card 
                    sx={{ 
                        cursor: 'pointer', 
                        border: state.zwavePinCode === null ? '2px solid' : '1px solid',
                        borderColor: state.zwavePinCode === null ? 'warning.main' : 'divider',
                        '&:hover': { bgcolor: 'action.hover' } 
                    }}
                    onClick={() => setState(prev => ({ ...prev, zwavePinCode: null }))}
                >
                    <CardContent>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                            <Wifi color={state.zwavePinCode === null ? 'warning' : 'disabled'} />
                            <Box sx={{ flex: 1 }}>
                                <Typography variant="subtitle1" sx={{ fontWeight: 'bold' }}>
                                    Insecure Inclusion
                                </Typography>
                                <Typography variant="body2" color="text.secondary">
                                    Uses default PIN code (00000) for faster setup
                                </Typography>
                                <Chip label="Insecure" color="warning" size="small" sx={{ mt: 1 }} />
                            </Box>
                        </Box>
                    </CardContent>
                </Card>
            </Box>

            {state.zwavePinCode !== null && (
                <Box sx={{ mb: 3 }}>
                    <Alert severity="info" sx={{ mb: 2 }}>
                        <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                            Where to find your PIN code:
                        </Typography>
                        <Typography variant="body2">
                            • Check the device label or sticker<br/>
                            • Look in the device manual or packaging<br/>
                            • Some devices display it on screen during pairing
                        </Typography>
                    </Alert>
                    
                    <TextField
                        fullWidth
                        label="Device PIN Code"
                        value={state.zwavePinCode || ''}
                        onChange={(e) => {
                            const value = e.target.value.replace(/\D/g, '').slice(0, 10);
                            setState(prev => ({ ...prev, zwavePinCode: value }));
                        }}
                        placeholder="Enter 5-digit code (e.g., 12345)"
                        helperText="Enter the PIN code from your Z-Wave device"
                        inputProps={{
                            maxLength: 10,
                            pattern: '[0-9]*',
                            inputMode: 'numeric'
                        }}
                    />
                </Box>
            )}

            {state.zwavePinCode === null && (
                <Alert severity="warning" sx={{ mb: 3 }}>
                    <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                        Insecure Inclusion Mode:
                    </Typography>
                    <Typography variant="body2">
                        Your device will be paired using the default PIN code (00000). This is faster but less secure than using your device's actual PIN code.
                    </Typography>
                </Alert>
            )}

            <Button
                variant="contained"
                color={state.zwavePinCode !== null ? 'success' : 'warning'}
                onClick={handleSimpleInclusion}
                disabled={state.loading || (state.zwavePinCode !== null && (!state.zwavePinCode || state.zwavePinCode.length < 5))}
                startIcon={state.loading ? <CircularProgress size={20} /> : (state.zwavePinCode !== null ? <Security /> : <Wifi />)}
                fullWidth
                size="large"
            >
                {state.loading ? 'Starting Discovery...' : (state.zwavePinCode !== null ? 'Start Secure Discovery' : 'Start Insecure Discovery')}
            </Button>

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderCurrentStep = () => {
        switch (state.step) {
            case OnboardingStep.ZWAVE_PIN_SETUP:
                return renderZWavePinSetup();
            case OnboardingStep.DEVICE_CREATION:
                return renderDeviceCreation();
            case OnboardingStep.DISCOVERY:
                return renderDiscovery();
            case OnboardingStep.NAMING:
                return renderNaming();
            case OnboardingStep.COMPLETE:
                return renderComplete();
            default:
                return null;
        }
    };

    return (
        <Box sx={{ maxWidth: 600, margin: '0 auto', p: 3 }}>
            <Typography variant="h4" gutterBottom sx={{ textAlign: 'center', mb: 4 }}>
                {protocol} User Guided Setup
            </Typography>

            <Stepper activeStep={getStepperIndex(state.step, protocol)} sx={{ mb: 4 }}>
                {steps.map((label) => (
                    <Step key={label}>
                        <StepLabel>{label}</StepLabel>
                    </Step>
                ))}
            </Stepper>

            <Card elevation={3}>
                <CardContent sx={{ p: 4 }}>
                    {renderCurrentStep()}
                </CardContent>
            </Card>

            <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 3 }}>
                <Button
                    variant="outlined"
                    onClick={onCancel}
                    disabled={state.loading}
                >
                    Cancel
                </Button>
                
                {state.step > (protocol === ProtocolType.ZWAVE ? OnboardingStep.ZWAVE_PIN_SETUP : OnboardingStep.DEVICE_CREATION) && state.step < OnboardingStep.COMPLETE && (
                    <Button
                        variant="outlined"
                        onClick={() => setState(prev => ({ 
                            ...prev, 
                            step: prev.step - 1,
                            error: null 
                        }))}
                        disabled={state.loading}
                    >
                        Back
                    </Button>
                )}
            </Box>
        </Box>
    );
};

export default SimpleDeviceOnboarding;
