import React, { useState, useEffect, useRef } from 'react';
import {
    Box,
    Card,
    CardContent,
    Typography,
    Button,
    Alert,
    Stepper,
    Step,
    StepLabel,
    CircularProgress,
    Chip,
    TextField
} from '@mui/material';
import {
    QrCode as QrCodeIcon,
    CheckCircle,
    Security
} from '@mui/icons-material';
import { ProtocolType } from '../../types/deviceTypes';
import { deviceService } from '../../services/deviceService';
import { HUB_CONFIG } from '../../constants';
import SimpleQRScanner from '../SimpleQRScanner';

interface QRCodeOnboardingProps {
    protocol: ProtocolType;
    onComplete: (managedThingId: string, deviceName?: string) => void;
    onCancel: () => void;
}

enum OnboardingStep {
    QR_SCAN = 0,
    CREATE_DEVICE = 1,
    DEVICE_ACTIVATION = 2,
    NAMING = 3,
    COMPLETE = 4
}

interface OnboardingState {
    step: OnboardingStep;
    scannedCode: string | null;
    managedThingId: string | null;
    deviceName: string;
    loading: boolean;
    error: string | null;
    activationStarted: boolean;
    pollingInterval: NodeJS.Timeout | null;
    deviceStatus: string | null;
}

const QRCodeOnboarding: React.FC<QRCodeOnboardingProps> = ({
    protocol,
    onComplete,
    onCancel
}) => {
    const [state, setState] = useState<OnboardingState>({
        step: OnboardingStep.QR_SCAN,
        scannedCode: null,
        managedThingId: null,
        deviceName: '',
        loading: false,
        error: null,
        activationStarted: false,
        pollingInterval: null,
        deviceStatus: null
    });

    // Cleanup polling interval on unmount
    useEffect(() => {
        return () => {
            if (state.pollingInterval) {
                clearInterval(state.pollingInterval);
            }
        };
    }, [state.pollingInterval]);

    const steps = [
        'Scan QR Code',
        'Create Managed Thing',
        'Device Activation',
        'Name Device',
        'Complete'
    ];

    const handleQRCodeScanned = async (decodedText: string, decodedResult: any) => {
        console.log('QR Code scanned:', decodedText);
        console.log('QR code length:', decodedText.length);
        
        // Store the full QR code text for Simple Setup
        setState(prev => ({ 
            ...prev, 
            scannedCode: decodedText, // Use full QR code for Simple Setup
            step: OnboardingStep.CREATE_DEVICE,
            loading: true,
            error: null 
        }));

        try {
            console.log(`Creating ${protocol} device with QR code (Simple Setup):`, decodedText.substring(0, 20) + '...');
            
            // Create managed thing directly with QR code (Simple Setup approach)
            const protocolType = protocol === ProtocolType.ZWAVE ? 'ZWAVE' : 'ZIGBEE';
            const result = await deviceService.createDeviceWithQRCode(decodedText, protocolType);
            
            console.log('Managed thing created with ID:', result.managedThingId);
            
            setState(prev => ({
                ...prev,
                managedThingId: result.managedThingId,
                step: OnboardingStep.DEVICE_ACTIVATION,
                loading: false,
                activationStarted: true
            }));

            // Start polling for device activation
            startPollingForActivation(result.managedThingId);

        } catch (error) {
            console.error('Error creating device with QR code:', error);
            setState(prev => ({
                ...prev,
                loading: false,
                error: error instanceof Error ? error.message : 'Failed to create device with QR code. Please ensure the QR code is valid and try again.'
            }));
        }
    };

    const startPollingForActivation = (managedThingId: string) => {
        const interval = setInterval(async () => {
            try {
                const activationStatus = await deviceService.checkDeviceActivationStatus(managedThingId);
                console.log('Activation status:', activationStatus);
                
                setState(prev => ({
                    ...prev,
                    deviceStatus: activationStatus.status
                }));

                // If device is activated, move to naming step
                if (activationStatus.isActivated) {
                    console.log('Device activated successfully!');
                    
                    // Stop polling
                    clearInterval(interval);
                    
                    // Send initial ReadState to populate endpoints
                    try {
                        console.log('Sending initial ReadState after successful activation...');
                        await deviceService.sendInitialReadStateAfterOnboarding(managedThingId);
                        console.log('Initial ReadState sent successfully');
                    } catch (error) {
                        console.warn('Failed to send initial ReadState (non-critical):', error);
                    }
                    
                    setState(prev => ({
                        ...prev,
                        step: OnboardingStep.NAMING,
                        loading: false,
                        pollingInterval: null
                    }));
                }
            } catch (error) {
                console.error('Error checking device activation status:', error);
            }
        }, 3000); // Poll every 3 seconds

        setState(prev => ({
            ...prev,
            pollingInterval: interval
        }));

        // Stop polling after 300 seconds (5 minutes)
        setTimeout(() => {
            clearInterval(interval);
            setState(prev => ({
                ...prev,
                pollingInterval: null,
                error: 'Device activation timed out after 5 minutes. The device may still activate - check your dashboard.'
            }));
            console.log('Activation polling timed out after 300 seconds');
        }, 300000);
    };

    const handleComplete = () => {
        if (state.managedThingId) {
            onComplete(state.managedThingId, state.deviceName.trim() || undefined);
        }
    };

    const handleNameSubmit = async () => {
        if (!state.deviceName.trim()) {
            setState(prev => ({ ...prev, error: 'Please enter a device name' }));
            return;
        }

        try {
            setState(prev => ({ ...prev, loading: true, error: null }));

            // The name will be handled when the device is added to the store
            console.log(`Device named: ${state.deviceName}`);
            
            setState(prev => ({
                ...prev,
                step: OnboardingStep.COMPLETE,
                loading: false
            }));
        } catch (error) {
            console.error('Error saving device name:', error);
            setState(prev => ({
                ...prev,
                loading: false,
                error: error instanceof Error ? error.message : 'Failed to save device name'
            }));
        }
    };

    const handleRetryQRScan = () => {
        setState(prev => ({
            ...prev,
            step: OnboardingStep.QR_SCAN,
            scannedCode: null,
            managedThingId: null,
            error: null,
            activationStarted: false,
            loading: false,
            deviceStatus: null
        }));
    };

    const renderQRScan = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Scan {protocol} Device QR Code
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Point your camera at the QR code on your {protocol} device for secure pairing using Simple Setup.
            </Typography>

            <Alert severity="info" sx={{ mb: 3 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                    Where to find the code:
                </Typography>
                <Typography variant="body2">
                    • Check the device label or sticker<br/>
                    • Look in the device manual or packaging<br/>
                    • Some devices display it on screen during setup<br/>
                    • {protocol === ProtocolType.ZWAVE ? 'Look for DSK, S2 QR code, or Data Matrix' : 'Look for install code QR code or Data Matrix'}<br/>
                    • Supports QR codes, Data Matrix, and other barcode formats
                </Typography>
            </Alert>

            {state.loading ? (
                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center', p: 4 }}>
                    <CircularProgress size={40} sx={{ mr: 2 }} />
                    <Typography>Processing scanned code...</Typography>
                </Box>
            ) : (
                <Box>
                    <Alert severity="info" sx={{ mb: 2 }}>
                        <Typography variant="body2">
                            <strong>Tips for better scanning:</strong><br/>
                            • Hold the code steady for 2-3 seconds<br/>
                            • Ensure good lighting<br/>
                            • Keep the code flat and unfolded<br/>
                            • Try different distances from the camera<br/>
                            • Works with QR codes, Data Matrix, and barcodes
                        </Typography>
                    </Alert>
                    
                    <SimpleQRScanner
                        onScanSuccess={(decodedText: string) => {
                            console.log('QR SCANNER CALLBACK TRIGGERED!');
                            console.log('Decoded text:', decodedText);
                            handleQRCodeScanned(decodedText, null);
                        }}
                        onScanError={(error: string) => {
                            // Suppress routine errors to reduce console noise
                            if (!error.includes('No QR code found') && 
                                !error.includes('NotFoundException') &&
                                !error.includes('No barcode or QR code detected')) {
                                console.warn('QR scan error:', error);
                            }
                        }}
                        width={600}
                        height={400}
                    />
                </Box>
            )}

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderCreateDevice = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Creating Managed Thing
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Creating a managed thing with your scanned QR code using Simple Setup.
            </Typography>

            <Alert severity="success" sx={{ mb: 3 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                    QR Code Scanned Successfully:
                </Typography>
                <Typography variant="body2">
                    <strong>Scanned Code:</strong> {state.scannedCode ? `${state.scannedCode.substring(0, 20)}...` : 'N/A'}
                </Typography>
            </Alert>

            <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center', p: 4 }}>
                <CircularProgress size={40} sx={{ mr: 2 }} />
                <Typography>Creating managed thing with QR code...</Typography>
            </Box>

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                    <Box sx={{ mt: 1 }}>
                        <Button size="small" onClick={handleRetryQRScan}>
                            Scan Different QR Code
                        </Button>
                    </Box>
                </Alert>
            )}
        </Box>
    );

    const renderDeviceActivation = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Device Activation
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Managed thing created successfully. Now plug in your {protocol} device and wait for it to activate.
            </Typography>

            <Alert severity="info" sx={{ mb: 3 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 'bold', mb: 1 }}>
                    Next Steps:
                </Typography>
                <Typography component="div">
                    1. Plug in your {protocol} device<br/>
                    2. The device will automatically connect using the QR code credentials<br/>
                    3. Wait for the device to become ACTIVATED (up to 5 minutes)<br/>
                    4. Device will appear on your dashboard once activated
                </Typography>
            </Alert>

            {state.managedThingId && (
                <Alert severity="success" sx={{ mb: 2 }}>
                    <Typography variant="body2">
                        <strong>Managed Thing ID:</strong> {state.managedThingId}<br/>
                        <strong>Current Status:</strong> {state.deviceStatus || 'Checking...'}
                    </Typography>
                </Alert>
            )}

            {state.activationStarted && (
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, mb: 2 }}>
                    <CircularProgress size={24} />
                    <Typography>
                        Waiting for device activation... Status: {state.deviceStatus || 'Checking...'}
                    </Typography>
                </Box>
            )}

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                    <Box sx={{ mt: 1 }}>
                        <Button size="small" onClick={handleRetryQRScan}>
                            Start Over
                        </Button>
                    </Box>
                </Alert>
            )}
        </Box>
    );

    const renderNaming = () => (
        <Box>
            <Typography variant="h6" gutterBottom>
                Name Your Device
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Give your {protocol} device a memorable name to make it easier to identify.
            </Typography>

            {state.managedThingId && (
                <Alert severity="success" sx={{ mb: 3 }}>
                    <Typography variant="body2">
                        <strong>Device ID:</strong> {state.managedThingId}<br/>
                        <strong>Status:</strong> {state.deviceStatus || 'ACTIVATED'}
                    </Typography>
                </Alert>
            )}

            <TextField
                fullWidth
                label="Device Name"
                value={state.deviceName}
                onChange={(e) => setState(prev => ({ ...prev, deviceName: e.target.value, error: null }))}
                placeholder={`My ${protocol} Device`}
                helperText="Enter a name that will help you identify this device"
                sx={{ mb: 3 }}
                onKeyPress={(e) => e.key === 'Enter' && handleNameSubmit()}
            />

            <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 3 }}>
                <Button
                    variant="outlined"
                    onClick={() => setState(prev => ({ ...prev, step: OnboardingStep.DEVICE_ACTIVATION }))}
                    disabled={state.loading}
                >
                    Back
                </Button>
                <Button
                    variant="contained"
                    onClick={handleNameSubmit}
                    disabled={state.loading || !state.deviceName.trim()}
                    startIcon={state.loading ? <CircularProgress size={20} /> : undefined}
                >
                    {state.loading ? 'Saving...' : 'Save Name'}
                </Button>
            </Box>

            {state.error && (
                <Alert severity="error" sx={{ mt: 2 }}>
                    {state.error}
                </Alert>
            )}
        </Box>
    );

    const renderComplete = () => (
        <Box sx={{ textAlign: 'center' }}>
            <CheckCircle color="success" sx={{ fontSize: 64, mb: 2 }} />
            <Typography variant="h6" gutterBottom>
                {protocol} Device Added Successfully!
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Your {protocol} device has been securely added to the network using Simple Setup.
            </Typography>

            {state.managedThingId && (
                <Alert severity="success" sx={{ mb: 3 }}>
                    <Typography variant="body2">
                        <strong>Device ID:</strong> {state.managedThingId}<br/>
                        <strong>Device Name:</strong> {state.deviceName || 'Unnamed Device'}<br/>
                        <strong>Security Method:</strong> QR Code Simple Setup<br/>
                        <strong>Status:</strong> {state.deviceStatus || 'ACTIVATED'}
                    </Typography>
                </Alert>
            )}

            <Alert severity="info" sx={{ mb: 3 }}>
                <Typography variant="body2">
                    The device has been successfully activated and is ready to use. 
                    It will appear on your dashboard and you can start controlling it immediately.
                </Typography>
            </Alert>

            <Button
                variant="contained"
                onClick={handleComplete}
                size="large"
            >
                Return to Dashboard
            </Button>
        </Box>
    );

    const renderCurrentStep = () => {
        switch (state.step) {
            case OnboardingStep.QR_SCAN:
                return renderQRScan();
            case OnboardingStep.CREATE_DEVICE:
                return renderCreateDevice();
            case OnboardingStep.DEVICE_ACTIVATION:
                return renderDeviceActivation();
            case OnboardingStep.NAMING:
                return renderNaming();
            case OnboardingStep.COMPLETE:
                return renderComplete();
            default:
                return null;
        }
    };

    return (
        <Box sx={{ maxWidth: 600, margin: '0 auto', p: 3 }}>
            <Typography variant="h4" gutterBottom sx={{ textAlign: 'center', mb: 4 }}>
                {protocol} Simple Setup
            </Typography>

            <Stepper activeStep={state.step} sx={{ mb: 4 }}>
                {steps.map((label) => (
                    <Step key={label}>
                        <StepLabel>{label}</StepLabel>
                    </Step>
                ))}
            </Stepper>

            <Card elevation={3}>
                <CardContent sx={{ p: 4 }}>
                    {renderCurrentStep()}
                </CardContent>
            </Card>

            <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 3 }}>
                <Button
                    variant="outlined"
                    onClick={onCancel}
                    disabled={state.loading}
                >
                    Cancel
                </Button>
                
                {(state.step === OnboardingStep.CREATE_DEVICE || state.step === OnboardingStep.DEVICE_ACTIVATION) && (
                    <Button
                        variant="outlined"
                        onClick={handleRetryQRScan}
                        disabled={state.loading}
                    >
                        Start Over
                    </Button>
                )}
            </Box>
        </Box>
    );
};

export default QRCodeOnboarding;
