import React, { useState } from 'react';
import {
    Box,
    Button,
    Typography,
    CircularProgress,
    Alert,
    Paper,
} from '@mui/material';
import { deviceService } from '../../services/deviceService';

interface ZigbeeDiscoveryProps {
    managedThingId: string;
}

export const ZigbeeDiscovery: React.FC<ZigbeeDiscoveryProps> = ({ managedThingId }) => {
    const [isDiscovering, setIsDiscovering] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handleStartDiscovery = async () => {
        try {
            setError(null);
            setIsDiscovering(true);
            await deviceService.startZigbeeDiscovery(managedThingId);
        } catch (error) {
            console.error('Error starting Zigbee discovery:', error);
            setError('Failed to start Zigbee device discovery');
        } finally {
            setIsDiscovering(false);
        }
    };

    return (
        <Paper sx={{ p: 2, mb: 2 }}>
            <Typography variant="h6" gutterBottom>
                Zigbee Device Discovery
            </Typography>
            <Typography variant="body2" color="text.secondary" paragraph>
                Start Zigbee device discovery to find and add new Zigbee devices to your network.
                The discovery will automatically stop after a timeout.
            </Typography>
            {error && (
                <Alert severity="error" sx={{ mb: 2 }}>
                    {error}
                </Alert>
            )}
            <Box sx={{ display: 'flex', gap: 2 }}>
                <Button
                    variant="contained"
                    color="primary"
                    onClick={handleStartDiscovery}
                    disabled={isDiscovering}
                    startIcon={isDiscovering ? <CircularProgress size={20} /> : null}
                >
                    Start Discovery
                </Button>
            </Box>
        </Paper>
    );
}; 