// src/components/devices/DeviceWidget.tsx

import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { 
    Card, 
    CardContent, 
    CardActions, 
    Typography, 
    Button, 
    Box, 
    Chip, 
    Stack, 
    IconButton,
    TextField,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogContentText,
    DialogActions
} from '@mui/material';
import { Edit as EditIcon, Check as CheckIcon, Close as CloseIcon } from '@mui/icons-material';
import { Device } from '../../types/deviceTypes';
import { getClusterComponent } from '../../clusters';
import { useDispatch } from 'react-redux';
import { deviceService } from '../../services/deviceService';
import { updateDeviceName } from '../../store/deviceSlice';
import { debugLogger } from '../../utils/debugLogger';

interface DeviceWidgetProps {
    device: Device;
}

const DeviceWidget: React.FC<DeviceWidgetProps> = ({ device }) => {
    const [isEditingName, setIsEditingName] = useState(false);
    const [editedName, setEditedName] = useState(device.Name || '');
    // Don't show name prompt automatically - user can manually edit name if needed
    const [showNamePrompt, setShowNamePrompt] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [isUpdating, setIsUpdating] = useState(false);
    const dispatch = useDispatch();

    // Update local state when device prop changes
    useEffect(() => {
        setEditedName(device.Name || '');
    }, [device.Name]);

    // Infer device type based on capabilities
    const inferDeviceType = () => {
        const capabilities = device.currentState?.Endpoints?.[0]?.capabilities || [];
        const hasOnOff = capabilities.some(cap => cap.id.includes('OnOff'));
        const hasDimming = capabilities.some(cap => cap.id.includes('LevelControl'));
        const hasColor = capabilities.some(cap => cap.id.includes('ColorControl'));
        const hasTemperature = capabilities.some(cap => cap.id.includes('TemperatureMeasurement') || cap.name.includes('Temperature'));
        const hasBooleanState = capabilities.some(cap => cap.id.includes('BooleanState') || cap.name.includes('Boolean'));
        const hasPowerSource = capabilities.some(cap => cap.id.includes('PowerSource') || cap.name.includes('Power Source'));

        // Light devices
        if (hasColor) return 'Color Light';
        if (hasDimming) return 'Dimmable Light';
        if (hasOnOff) return 'Simple Switch';
        
        // Sensor devices
        if (hasTemperature && hasBooleanState) return 'Multi Sensor';
        if (hasTemperature) return 'Temperature Sensor';
        if (hasBooleanState) return 'Contact Sensor';
        if (hasPowerSource && !hasOnOff) return 'Battery Sensor';
        
        return 'Unknown Device';
    };

    const deviceType = inferDeviceType();

    const handleEditName = () => {
        setEditedName(device.Name || '');
        setIsEditingName(true);
    };

    const handleSaveName = async () => {
        if (!editedName.trim()) {
            setError('Device name cannot be empty');
            return;
        }

        try {
            setIsUpdating(true);
            setError(null);
            
            // Update the device name in the backend
            await deviceService.updateDeviceName(device.ManagedThingId, editedName.trim());
            
            // Update Redux store
            dispatch(updateDeviceName({
                ManagedThingId: device.ManagedThingId,
                name: editedName.trim()
            }));
            
            // Exit edit mode
            setIsEditingName(false);
            setShowNamePrompt(false);
        } catch (err) {
            console.error('Failed to update device name:', err);
            setError('Failed to update device name');
            setEditedName(device.Name || '');
        } finally {
            setIsUpdating(false);
        }
    };

    debugLogger.log('DeviceWidget rendering:', {
        device,
        name: device.Name,
        protocol: device.Protocol,
        type: deviceType,
        currentState: device.currentState,
    });

    return (
        <>
            {/* Name Prompt Dialog for unnamed devices */}
            <Dialog 
                open={showNamePrompt} 
                onClose={() => setShowNamePrompt(false)}
                aria-labelledby="name-dialog-title"
            >
                <DialogTitle id="name-dialog-title">Name Your Device</DialogTitle>
                <DialogContent>
                    <DialogContentText>
                        Please enter a name for your device to make it easier to identify.
                    </DialogContentText>
                    <TextField
                        autoFocus
                        margin="dense"
                        id="name"
                        label="Device Name"
                        type="text"
                        fullWidth
                        variant="outlined"
                        value={editedName}
                        onChange={(e) => setEditedName(e.target.value)}
                        error={!!error}
                        helperText={error}
                        onKeyPress={(e) => e.key === 'Enter' && handleSaveName()}
                        sx={{ mt: 2 }}
                    />
                </DialogContent>
                <DialogActions>
                    <Button 
                        onClick={() => {
                            setShowNamePrompt(false);
                            // Use a default name if user cancels
                            if (!editedName.trim()) {
                                const defaultName = `Device ${device.ManagedThingId.substring(0, 4)}`;
                                setEditedName(defaultName);
                                handleSaveName();
                            }
                        }}
                    >
                        Cancel
                    </Button>
                    <Button 
                        onClick={handleSaveName}
                        variant="contained"
                        disabled={!editedName.trim() || isUpdating}
                    >
                        Save
                    </Button>
                </DialogActions>
            </Dialog>

            <Card sx={{ minWidth: 275, m: 2 }}>
                <CardContent>
                    <Stack spacing={2}>
                        {/* Device Name, Type and Protocol */}
                        <Box sx={{ 
                            display: 'flex', 
                            flexDirection: 'column',
                            gap: 1
                        }}>
                            <Box sx={{
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'space-between'
                            }}>
                                {isEditingName ? (
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <TextField
                                            value={editedName}
                                            onChange={(e) => setEditedName(e.target.value)}
                                            onKeyPress={(e) => e.key === 'Enter' && handleSaveName()}
                                            size="small"
                                            autoFocus
                                            error={!!error}
                                            helperText={error}
                                        />
                                        <IconButton onClick={handleSaveName} color="primary" size="small" disabled={isUpdating}>
                                            <CheckIcon />
                                        </IconButton>
                                        <IconButton 
                                            onClick={() => {
                                                setIsEditingName(false);
                                                setEditedName(device.Name || '');
                                                setError(null);
                                            }} 
                                            size="small"
                                        >
                                            <CloseIcon />
                                        </IconButton>
                                    </Box>
                                ) : (
                                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                        <Typography variant="h5" component="div">
                                            {device.Name || 'Unnamed Device'}
                                        </Typography>
                                        <IconButton onClick={handleEditName} size="small">
                                            <EditIcon />
                                        </IconButton>
                                    </Box>
                                )}
                                <Chip
                                    label={device.Protocol}
                                    size="small"
                                    color="primary"
                                    variant="outlined"
                                />
                            </Box>
                            <Typography variant="body2" color="text.secondary">
                                {deviceType}
                            </Typography>
                        </Box>

                        {/* Capabilities - Use CapabilityReport (what device CAN do) not currentState (current values) */}
                        {device.CapabilityReport?.endpoints?.map((capabilityEndpoint) => {
                            debugLogger.log('Processing capability endpoint:', {
                                capabilityEndpoint,
                                endpointId: capabilityEndpoint.id,
                                capabilities: capabilityEndpoint.capabilities
                            });
                            
                            return (
                                <Box key={capabilityEndpoint.id}>
                                    {capabilityEndpoint.capabilities?.map((capabilityDef) => {
                                        debugLogger.log('Processing capability definition:', {
                                            capabilityDef,
                                            id: capabilityDef.id,
                                            name: capabilityDef.name
                                        });
                                        
                                        // Find matching state data for this capability (if available)
                                        const stateEndpoint = device.currentState?.Endpoints?.find(
                                            ep => ep.endpointId === capabilityEndpoint.id
                                        );
                                        const stateCapability = stateEndpoint?.capabilities?.find(
                                            cap => cap.id === capabilityDef.id
                                        );
                                        
                                        // Create capability object with definition + state
                                        const capability = {
                                            id: capabilityDef.id,
                                            name: capabilityDef.name,
                                            version: capabilityDef.version,
                                            actions: capabilityDef.actions,
                                            events: capabilityDef.events,
                                            properties: stateCapability?.properties || [] // Use state properties if available, empty array if not
                                        };
                                        
                                        debugLogger.log('Created merged capability:', {
                                            capability,
                                            hasStateData: !!stateCapability,
                                            propertiesCount: capability.properties.length
                                        });
                                        
                                        try {
                                            const ClusterComponent = getClusterComponent(capability);
                                            debugLogger.log('Got cluster component:', {
                                                hasComponent: !!ClusterComponent,
                                                hasRenderWidget: ClusterComponent?.renderWidget,
                                                component: ClusterComponent
                                            });
                                            
                                            if (ClusterComponent?.renderWidget) {
                                                debugLogger.log('Rendering widget for capability:', capability.id);
                                                return (
                                                    <Box key={capability.id} sx={{ mb: 2 }}>
                                                        {ClusterComponent.renderWidget({ 
                                                            capability, 
                                                            deviceId: device.ManagedThingId 
                                                        })}
                                                    </Box>
                                                );
                                            }
                                            debugLogger.log('No cluster component found for capability:', capability.id);
                                            return null;
                                        } catch (error) {
                                            console.error('Error rendering capability widget:', {
                                                capabilityId: capability.id,
                                                error: error instanceof Error ? error.message : 'Unknown error',
                                                stack: error instanceof Error ? error.stack : undefined
                                            });
                                            return (
                                                <Box key={capability.id} sx={{ mb: 2, p: 2, bgcolor: 'error.light', borderRadius: 1 }}>
                                                    <Typography variant="body2" color="error.contrastText">
                                                        Error loading {capability.name} widget
                                                    </Typography>
                                                </Box>
                                            );
                                        }
                                    })}
                                </Box>
                            );
                        })}
                    </Stack>
                </CardContent>
                <CardActions>
                    <Button
                        size="small"
                        component={Link}
                        to={`/device/${device.ManagedThingId}`}
                    >
                        View Details
                    </Button>
                </CardActions>
            </Card>
        </>
    );
};

export default DeviceWidget;
