import React, { useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import {
    Dialog,
    DialogTitle,
    DialogContent,
    DialogActions,
    TextField,
    Button,
    Typography,
    Box,
    CircularProgress,
    Alert
} from '@mui/material';
import { deviceService } from '../../services/deviceService';
import { HUB_CONFIG } from '../../constants';
import { addDevice } from '../../store/deviceSlice';
import { Device } from '../../types/deviceTypes';
import { RootState } from '../../store/store';

interface DeviceInitializationProps {
    hubId: string;
    onComplete: () => void;
}

const DeviceInitialization: React.FC<DeviceInitializationProps> = ({ hubId, onComplete }) => {
    const dispatch = useDispatch();
    const existingDevices = useSelector((state: RootState) => state.devices.devices);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [newDevices, setNewDevices] = useState<Device[]>([]);
    const [currentDeviceIndex, setCurrentDeviceIndex] = useState(0);
    const [deviceName, setDeviceName] = useState('');
    const [showNameDialog, setShowNameDialog] = useState(false);

    useEffect(() => {
        const initializeDevices = async () => {
            try {
                setLoading(true);
                setError(null);

                const deviceIds = await deviceService.listManagedThings(HUB_CONFIG.MANAGED_THING_ID);
                
                // Filter out devices that already exist in the store
                const newDeviceIds = deviceIds.filter(id => 
                    !existingDevices.some(existing => existing.ManagedThingId === id)
                );

                if (newDeviceIds.length === 0) {
                    // No new devices, complete initialization
                    onComplete();
                    return;
                }

                // Get full details for each new device
                const devicesWithDetails = await Promise.all(
                    newDeviceIds.map(id => deviceService.getDeviceWithDetails(id))
                );

                setNewDevices(devicesWithDetails);
                setShowNameDialog(true);
            } catch (error) {
                console.error('Error initializing devices:', error);
                setError('Failed to initialize devices');
            } finally {
                setLoading(false);
            }
        };

        initializeDevices();
    }, [existingDevices, onComplete]);

    const handleNameSubmit = () => {
        if (!deviceName.trim()) return;

        const currentDevice = newDevices[currentDeviceIndex];
        const deviceWithName = {
            ...currentDevice,
            Name: deviceName.trim()
        };

        // Add device to store
        dispatch(addDevice(deviceWithName));

        // Reset form and move to next device
        setDeviceName('');
        
        // Check if there are more devices to process
        if (currentDeviceIndex < newDevices.length - 1) {
            setCurrentDeviceIndex(prev => prev + 1);
        } else {
            setShowNameDialog(false);
            onComplete();
        }
    };

    if (loading) {
        return (
            <Box sx={{ display: 'flex', justifyContent: 'center', p: 3 }}>
                <CircularProgress />
            </Box>
        );
    }

    if (error) {
        return (
            <Alert severity="error" sx={{ m: 2 }}>
                {error}
            </Alert>
        );
    }

    return (
        <Dialog open={showNameDialog} onClose={() => {}}>
            <DialogTitle>Name Your Device</DialogTitle>
            <DialogContent>
                <Typography variant="body1" sx={{ mb: 2 }}>
                    Please provide a name for your device:
                </Typography>
                <TextField
                    autoFocus
                    margin="dense"
                    label="Device Name"
                    fullWidth
                    value={deviceName}
                    onChange={(e) => setDeviceName(e.target.value)}
                    onKeyPress={(e) => {
                        if (e.key === 'Enter') {
                            handleNameSubmit();
                        }
                    }}
                />
            </DialogContent>
            <DialogActions>
                <Button onClick={handleNameSubmit} variant="contained">
                    {currentDeviceIndex === newDevices.length - 1 ? 'Finish' : 'Next'}
                </Button>
            </DialogActions>
        </Dialog>
    );
};

export default DeviceInitialization; 