// src/components/devices/DeviceDetail.tsx

import React, { useEffect, useState } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useSelector, useDispatch } from 'react-redux';
import {
    Card,
    CardContent,
    Typography,
    Button,
    Box,
    CircularProgress,
    Chip,
    IconButton,
    Alert,
    Stack,
    Switch,
    Divider,
    Paper,
    TextField,
    Dialog,
    DialogTitle,
    DialogContent,
    DialogContentText,
    DialogActions,
} from '@mui/material';
import * as Icons from '@mui/icons-material';
import { RootState } from '../../store/store';
import { deviceService } from '../../services/deviceService';
import { removeDevice, updateDeviceState, addDevice, updateDeviceName } from '../../store/deviceSlice';
import { subscribeToDevice, unsubscribeFromDevice } from '../../store/subscriptionSlice';
import { deviceStatePollingService } from '../../services/deviceStatePollingService';
import { deviceMonitorService } from '../../services/deviceMonitorService';
import { getClusterComponent } from '../../clusters';
import {
    Device,
    DeviceState,
    Capability,
    OnOffCapability,
    Endpoint,
    isOnOffCapability,
    MATTER_DEVICE_TYPES,
    MATTER_CAPABILITIES,
    ProtocolType,
    EndpointDefinition
} from '../../types/deviceTypes';

const DeviceDetail: React.FC = () => {
    const { id } = useParams<{ id: string }>();
    const navigate = useNavigate();
    const dispatch = useDispatch();

    // Get device from Redux store
    const storeDevice = useSelector((state: RootState) => 
        state.devices.devices.find(d => d.ManagedThingId === id)
    );

    const [device, setDevice] = useState<Device | null>(null);
    const [deviceState, setDeviceState] = useState<DeviceState | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [isEditingName, setIsEditingName] = useState(false);
    const [editedName, setEditedName] = useState('');
    const [showNamePrompt, setShowNamePrompt] = useState(false);
    const [showZWaveDeleteDialog, setShowZWaveDeleteDialog] = useState(false);

    // Infer device type based on capabilities
    const inferDeviceType = (deviceState: DeviceState | null) => {
        if (!deviceState?.Endpoints?.[0]?.capabilities) return 'Unknown Device';
        
        const capabilities = deviceState.Endpoints[0].capabilities;
        const hasOnOff = capabilities.some(cap => cap.id.includes('OnOff'));
        const hasDimming = capabilities.some(cap => cap.id.includes('LevelControl'));
        const hasColor = capabilities.some(cap => cap.id.includes('ColorControl'));

        if (hasColor) return 'Color Light';
        if (hasDimming) return 'Dimmable Light';
        if (hasOnOff) return 'Simple Switch';
        return 'Unknown Device';
    };

    // Check if device needs a name when it's first loaded
    useEffect(() => {
        if (device && (!device.Name || device.Name === 'Unnamed Device')) {
            console.log('Device needs a name:', device.ManagedThingId);
            setShowNamePrompt(true);
            setEditedName('');  // Clear the name to force user input
        }
    }, [device?.ManagedThingId]);

    // Update local state when store device changes - CRITICAL for UI updates
    useEffect(() => {
        if (storeDevice) {
            console.log('DeviceDetail - Store device updated:', {
                deviceId: storeDevice.ManagedThingId,
                name: storeDevice.Name,
                hasCurrentState: !!storeDevice.currentState,
                currentState: storeDevice.currentState
            });
            setDevice(storeDevice);
            if (storeDevice.currentState) {
                console.log('DeviceDetail - Updating device state from store for UI refresh:', {
                    deviceId: storeDevice.ManagedThingId,
                    state: storeDevice.currentState,
                    onOffValue: storeDevice.currentState.Endpoints?.[0]?.capabilities?.find(c => c.name === 'On/Off' || c.name === 'OnOff')?.properties?.find(p => p.name === 'OnOff')?.value?.propertyValue
                });
                setDeviceState(storeDevice.currentState);
            }
        }
    }, [storeDevice, storeDevice?.currentState]); // Add currentState as dependency to ensure UI updates

    // Debug effect to track state changes
    useEffect(() => {
        console.log('DeviceDetail - State Update:', {
            deviceId: device?.ManagedThingId,
            deviceName: device?.Name,
            hasDeviceState: !!deviceState,
            deviceState: deviceState,
            timestamp: new Date().toISOString()
        });
    }, [device, deviceState]);

    // Gentle focused device polling - only poll the device you're viewing
    useEffect(() => {
        if (id) {
            console.log(`DeviceDetail mounted - starting gentle polling for focused device: ${id}`);
            
            const initializeGentlePolling = async () => {
                try {
                    // Start gentle polling for only this device
                    await deviceStatePollingService.setPollingMode('device-detail', id);
                    console.log(`Gentle polling started for focused device: ${id} (every 30 seconds)`);
                } catch (error) {
                    console.error('Error starting gentle polling:', error);
                }
            };
            
            initializeGentlePolling();
        }
        
        return () => {
            console.log('DeviceDetail unmounting - stopping focused device polling');
            deviceStatePollingService.setPollingMode('stopped');
            deviceMonitorService.setMonitoringMode('stopped');
        };
    }, [id]);

    // Subscribe to device updates
    useEffect(() => {
        if (id) {
            console.log('DeviceDetail - Subscribing to device updates:', id);
            const handleStateUpdate = (state: DeviceState) => {
                console.log('DeviceDetail - Received state update:', {
                    deviceId: id,
                    state,
                    timestamp: new Date().toISOString()
                });
                setDeviceState(state);
            };

            deviceStatePollingService.subscribeToDeviceUpdates(id, handleStateUpdate);

            return () => {
                console.log('DeviceDetail - Unsubscribing from device updates:', id);
                deviceStatePollingService.unsubscribeFromDeviceUpdates(id, handleStateUpdate);
            };
        }
    }, [id]);

    useEffect(() => {
        const fetchDeviceDetails = async () => {
            if (!id) {
                setError('Device ID is required');
                setLoading(false);
                return;
            }

            try {
                setLoading(true);
                setError(null);

                // Use store device if available
                if (storeDevice) {
                    setDevice(storeDevice);
                    if (storeDevice.currentState) {
                        setDeviceState(storeDevice.currentState);
                    }
                } else {
                    // Fallback to API if not in store
                    const deviceDetails = await deviceService.getDeviceWithDetails(id);
                    console.log('Fetched device details:', deviceDetails);
                    setDevice(deviceDetails);
                    if (deviceDetails.currentState) {
                        setDeviceState(deviceDetails.currentState);
                    }
                }

            } catch (err) {
                console.error('Error fetching device details:', err);
                setError(err instanceof Error ? err.message : 'Failed to fetch device details');
            } finally {
                setLoading(false);
            }
        };

        fetchDeviceDetails();
    }, [id, storeDevice]);

    const handleEditName = () => {
        if (!device) return;
        setEditedName(device.Name);
        setIsEditingName(true);
    };

    useEffect(() => {
        if (device) {
            setEditedName(device.Name || '');
        }
    }, [device]);

    const handleSaveName = async () => {
        if (!device || !editedName.trim()) {
            // Show error if name is empty
            setError('Device name cannot be empty');
            return;
        }

        try {
            setLoading(true);
            setError(null);

            // First, update the device name in the backend
            await deviceService.updateDeviceName(device.ManagedThingId, editedName.trim());

            // Then update name in Redux store
            dispatch(updateDeviceName({
                ManagedThingId: device.ManagedThingId,
                name: editedName.trim()
            }));
            
            // Update local state
            setDevice({
                ...device,
                Name: editedName.trim()
            });

            // Exit edit mode
            setIsEditingName(false);
            
            // Clear name prompt if it was showing
            setShowNamePrompt(false);

        } catch (err) {
            console.error('Failed to update device name:', err);
            setError('Failed to update device name');
            // Revert the name in local state
            setEditedName(device.Name || '');
        } finally {
            setLoading(false);
        }
    };

    const handleDelete = async () => {
        if (!device) return;

        try {
            setLoading(true);
            await deviceService.deleteDevice(device.ManagedThingId);
            dispatch(removeDevice(device.ManagedThingId));
            navigate('/');
        } catch (err) {
            setError('Failed to delete device');
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <Box sx={{ display: 'flex', justifyContent: 'center', p: 3 }}>
                <CircularProgress />
            </Box>
        );
    }

    if (error) {
        return (
            <Alert severity="error" sx={{ m: 2 }}>
                {error}
            </Alert>
        );
    }

    if (!device) {
        return (
            <Alert severity="warning" sx={{ m: 2 }}>
                No device found
            </Alert>
        );
    }

    return (
        <Box sx={{ p: 3 }}>
            {/* Name Prompt Dialog for unnamed devices */}
            <Dialog 
                open={showNamePrompt} 
                onClose={() => setShowNamePrompt(false)}
                aria-labelledby="name-dialog-title"
            >
                <DialogTitle id="name-dialog-title">Name Your Device</DialogTitle>
                <DialogContent>
                    <DialogContentText>
                        Please enter a name for your device to make it easier to identify.
                    </DialogContentText>
                    <TextField
                        autoFocus
                        margin="dense"
                        id="name"
                        label="Device Name"
                        type="text"
                        fullWidth
                        variant="outlined"
                        value={editedName}
                        onChange={(e) => setEditedName(e.target.value)}
                        onKeyPress={(e) => e.key === 'Enter' && handleSaveName()}
                        sx={{ mt: 2 }}
                    />
                </DialogContent>
                <DialogActions>
                    <Button 
                        onClick={() => {
                            setShowNamePrompt(false);
                            // Use a default name if user cancels
                            if (!editedName && device) {
                                const defaultName = `Device ${device.ManagedThingId.substring(0, 4)}`;
                                setEditedName(defaultName);
                                dispatch(updateDeviceName({
                                    ManagedThingId: device.ManagedThingId,
                                    name: defaultName
                                }));
                            }
                        }}
                    >
                        Cancel
                    </Button>
                    <Button 
                        onClick={handleSaveName}
                        variant="contained"
                        disabled={!editedName.trim()}
                    >
                        Save
                    </Button>
                </DialogActions>
            </Dialog>

            {/* Z-Wave Delete Warning Dialog */}
            <Dialog 
                open={showZWaveDeleteDialog} 
                onClose={() => setShowZWaveDeleteDialog(false)}
                aria-labelledby="zwave-delete-dialog-title"
                maxWidth="sm"
                fullWidth
            >
                <DialogTitle id="zwave-delete-dialog-title">
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                        <Icons.Warning color="warning" />
                        Z-Wave Device Deletion
                    </Box>
                </DialogTitle>
                <DialogContent>
                    <DialogContentText sx={{ mb: 2 }}>
                        <strong>Important:</strong> To complete the deletion of this Z-Wave device, you will need to manually put the device into pairing mode.
                    </DialogContentText>
                    <DialogContentText sx={{ mb: 2 }}>
                        <strong>Steps to complete deletion:</strong>
                    </DialogContentText>
                    <Box component="ol" sx={{ pl: 2, mb: 2 }}>
                        <Box component="li" sx={{ mb: 1 }}>
                            Click "Delete Device" below to start the deletion process
                        </Box>
                        <Box component="li" sx={{ mb: 1 }}>
                            Physically access your Z-Wave device
                        </Box>
                        <Box component="li" sx={{ mb: 1 }}>
                            Put the device into pairing/exclusion mode (refer to device manual)
                        </Box>
                        <Box component="li">
                            The device will be fully removed from the network
                        </Box>
                    </Box>
                    <Alert severity="info" sx={{ mt: 2 }}>
                        The deletion process will begin immediately, but the device will remain partially connected until you manually put it in pairing mode.
                    </Alert>
                </DialogContent>
                <DialogActions>
                    <Button 
                        onClick={() => setShowZWaveDeleteDialog(false)}
                    >
                        Cancel
                    </Button>
                    <Button 
                        onClick={() => {
                            setShowZWaveDeleteDialog(false);
                            handleDelete();
                        }}
                        variant="contained"
                        color="error"
                        startIcon={<Icons.Delete />}
                    >
                        Delete Device
                    </Button>
                </DialogActions>
            </Dialog>

            {/* Header with Back Button */}
            <Box sx={{ mb: 3, display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                <Button
                    startIcon={<Icons.ArrowBack />}
                    onClick={() => navigate('/')}
                    variant="outlined"
                >
                    Back to Dashboard
                </Button>
            </Box>

            {/* Device Name Section */}
            <Box sx={{ mb: 3, display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                    {isEditingName ? (
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <TextField
                                value={editedName}
                                onChange={(e) => setEditedName(e.target.value)}
                                onKeyPress={(e) => e.key === 'Enter' && handleSaveName()}
                                size="small"
                                autoFocus
                            />
                            <IconButton onClick={handleSaveName} color="primary" size="small">
                                <Icons.Check />
                            </IconButton>
                            <IconButton onClick={() => {
                                setIsEditingName(false);
                                setEditedName(device.Name || '');
                            }} size="small">
                                <Icons.Close />
                            </IconButton>
                        </Box>
                    ) : (
                        <>
                            <Typography variant="h5" component="h1">
                                {device.Name || 'Unnamed Device'}
                            </Typography>
                            <Button 
                                startIcon={<Icons.Edit />}
                                variant="outlined"
                                size="small"
                                onClick={handleEditName}
                            >
                                Edit Name
                            </Button>
                        </>
                    )}
                </Box>
            </Box>

            <Card>
                <CardContent>
                    <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
                        <Stack direction="row" spacing={1} alignItems="center">
                            <Chip
                                label={device?.Protocol}
                                size="small"
                                color={device?.Protocol === ProtocolType.ZIGBEE ? "primary" : "secondary"}
                            />
                            <Chip
                                label={device?.CapabilityReport.nodeId}
                                size="small"
                                variant="outlined"
                            />
                        </Stack>
                        <Button
                            color="error"
                            variant="outlined"
                            startIcon={<Icons.Delete />}
                            onClick={() => {
                                // Show Z-Wave warning dialog for Z-Wave devices, otherwise delete directly
                                if (device?.Protocol === ProtocolType.ZWAVE) {
                                    setShowZWaveDeleteDialog(true);
                                } else {
                                    handleDelete();
                                }
                            }}
                            disabled={loading}
                            size="small"
                        >
                            Delete Device
                        </Button>
                    </Box>

                    <Divider sx={{ my: 2 }} />

                    {/* Device Capabilities Section */}
                    <Box sx={{ mb: 3 }}>
                        <Typography variant="h6" gutterBottom>
                            Device Capabilities
                        </Typography>
                        <Paper variant="outlined" sx={{ p: 2 }}>
                            <Stack spacing={2}>
                                <Box>
                                    <Typography variant="subtitle2" color="text.secondary">
                                        Node ID
                                    </Typography>
                                    <Typography variant="body1">
                                        {device?.CapabilityReport.nodeId}
                                    </Typography>
                                </Box>
                                <Box>
                                    <Typography variant="subtitle2" color="text.secondary">
                                        Device Type
                                    </Typography>
                                    <Typography variant="body1">
                                        {inferDeviceType(deviceState)}
                                    </Typography>
                                </Box>
                                {device?.CapabilityReport.endpoints && (
                                    <Box>
                                        <Typography variant="subtitle2" color="text.secondary" gutterBottom>
                                            Endpoints
                                        </Typography>
                                        <Stack spacing={2}>
                                            {device.CapabilityReport.endpoints.map((endpoint: EndpointDefinition) => (
                                                <Box key={endpoint.id}>
                                                    <Typography variant="subtitle2">
                                                        Endpoint {endpoint.id}
                                                    </Typography>
                                                    {endpoint.deviceTypes && endpoint.deviceTypes.length > 0 && (
                                                        <Box sx={{ mt: 1 }}>
                                                            {endpoint.deviceTypes.map((type: string) => (
                                                                <Chip
                                                                    key={type}
                                                                    label={type}
                                                                    size="small"
                                                                    sx={{ mr: 1 }}
                                                                />
                                                            ))}
                                                        </Box>
                                                    )}
                                                </Box>
                                            ))}
                                        </Stack>
                                    </Box>
                                )}
                            </Stack>
                        </Paper>
                    </Box>

                    {/* Device State Section */}
                    <Box>
                        <Typography variant="h6" gutterBottom>
                            Device State
                        </Typography>
                        {loading ? (
                            <Box sx={{ display: 'flex', justifyContent: 'center', my: 4 }}>
                                <CircularProgress />
                            </Box>
                        ) : deviceState ? (
                            <Paper variant="outlined" sx={{ p: 2 }}>
                                <Stack spacing={2}>
                                    {deviceState.Endpoints.map((endpoint) => (
                                        <EndpointCard
                                            key={endpoint.endpointId}
                                            endpoint={endpoint}
                                            deviceId={device.ManagedThingId}
                                        />
                                    ))}
                                </Stack>
                            </Paper>
                        ) : (
                            <Alert severity="info">No device state available</Alert>
                        )}
                    </Box>



                    <Card sx={{ mb: 3 }}>
                        <CardContent>
                            <Typography variant="h6" gutterBottom>
                                Device Information
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                ID: {device.ManagedThingId}
                            </Typography>
                            <Typography variant="body2" color="text.secondary">
                                Type: {inferDeviceType(deviceState)}
                            </Typography>
                        </CardContent>
                    </Card>
                </CardContent>
            </Card>
        </Box>
    );
};

interface EndpointCardProps {
    endpoint: {
        endpointId: string;
        deviceTypes?: string[];
        capabilities: Capability[];
    };
    deviceId: string;
}

const EndpointCard: React.FC<EndpointCardProps> = ({ endpoint, deviceId }) => {
    return (
        <Box>
            <Typography variant="subtitle1" gutterBottom>
                Endpoint {endpoint.endpointId}
            </Typography>
            <Stack spacing={2}>
                {endpoint.capabilities.map((capability) => (
                    <CapabilityCard
                        key={capability.id}
                        capability={capability}
                        deviceId={deviceId}
                    />
                ))}
            </Stack>
        </Box>
    );
};

interface CapabilityCardProps {
    capability: Capability;
    deviceId: string;
}

const CapabilityCard: React.FC<CapabilityCardProps> = ({ capability, deviceId }) => {
    const ClusterComponent = getClusterComponent(capability);

    if (ClusterComponent) {
        return ClusterComponent.renderDetails({ capability, deviceId });
    }

    return (
        <Box>
            <Typography variant="subtitle2" color="text.secondary">
                {capability.name} (v{capability.version})
            </Typography>
            <Box sx={{ mt: 1 }}>
                {capability.properties.map((prop) => (
                    <Box key={prop.name} sx={{ mb: 1 }}>
                        <Typography variant="body2" color="text.secondary">
                            {prop.name}
                        </Typography>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <Typography variant="body1">
                                {prop.value?.propertyValue?.toString() ?? 'N/A'}
                            </Typography>
                            {prop.value?.lastChangedAt && (
                                <Typography variant="caption" color="text.secondary">
                                    (Last updated: {new Date(prop.value.lastChangedAt).toLocaleString()})
                                </Typography>
                            )}
                        </Box>
                    </Box>
                ))}
            </Box>
            {capability.actions && capability.actions.length > 0 && (
                <Box sx={{ mt: 1 }}>
                    <Typography variant="body2" color="text.secondary">
                        Available Actions:
                    </Typography>
                    <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1, mt: 0.5 }}>
                        {capability.actions.map((action) => (
                            <Chip
                                key={action}
                                label={action}
                                size="small"
                                variant="outlined"
                            />
                        ))}
                    </Box>
                </Box>
            )}
        </Box>
    );
};


export default DeviceDetail;
