import React, { useState, useEffect } from 'react';
import {
    Box,
    Table,
    TableBody,
    TableCell,
    TableContainer,
    TableHead,
    TableRow,
    Typography,
    CircularProgress,
    Alert
} from '@mui/material';
import { deviceService } from '../../services/deviceService';
import { Device } from '../../types/deviceTypes';
import { HUB_CONFIG } from '../../constants';

export const DeviceList: React.FC = () => {
    const [devices, setDevices] = useState<Device[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        const fetchDevices = async () => {
            try {
                const deviceIds = await deviceService.listManagedThings(HUB_CONFIG.MANAGED_THING_ID);
                // Fetch full device details for each ID
                const devicesWithDetails = await Promise.all(
                    deviceIds.map(id => deviceService.getDevice(id))
                );
                setDevices(devicesWithDetails);
            } catch (err) {
                console.error('Error fetching devices:', err);
                setError('Failed to fetch devices');
            } finally {
                setLoading(false);
            }
        };

        fetchDevices();
    }, []);

    if (loading) {
        return (
            <Box sx={{ display: 'flex', justifyContent: 'center', p: 3 }}>
                <CircularProgress />
            </Box>
        );
    }

    if (error) {
        return (
            <Alert severity="error" sx={{ m: 2 }}>
                {error}
            </Alert>
        );
    }

    return (
        <>
            <Typography variant="h6" gutterBottom>
                Devices
            </Typography>
            <TableContainer>
                <Table>
                    <TableHead>
                        <TableRow>
                            <TableCell>Name</TableCell>
                            <TableCell>Protocol</TableCell>
                            <TableCell>Status</TableCell>
                        </TableRow>
                    </TableHead>
                    <TableBody>
                        {devices.map((device) => (
                            <TableRow key={device.ManagedThingId}>
                                <TableCell>{device.Name}</TableCell>
                                <TableCell>{device.Protocol}</TableCell>
                                <TableCell>Active</TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </TableContainer>
        </>
    );
}; 