import React, { useState } from 'react';
import {
    Box,
    TextField,
    Button,
    Typography,
    Paper,
    CircularProgress,
    Alert
} from '@mui/material';
import { deviceService } from '../services/deviceService';

interface ZWaveOnboardingProps {
    hubId?: string;
    onSuccess?: () => void;
    onError?: (error: Error) => void;
}

export const ZWaveOnboarding: React.FC<ZWaveOnboardingProps> = ({ hubId, onSuccess, onError }) => {
    const [pinCode, setPinCode] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handlePinCodeChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value.replace(/\D/g, '').slice(0, 5);
        setPinCode(value);
        setError(null);
    };

    const handleSubmit = async () => {
        if (pinCode.length !== 5) {
            setError('Please enter a valid 5-digit PIN code');
            return;
        }

        setIsLoading(true);
        setError(null);

        try {
            console.log('Starting Z-Wave device discovery with PIN:', pinCode);
            
            // Start device discovery with Z-Wave PIN code
            await deviceService.startDeviceDiscovery({
                discoveryType: 'ZWAVE',
                managedThingId: hubId,
                authenticationMaterial: pinCode,
                authenticationMaterialType: 'ZWAVE_INSTALL_CODE'
            });

            console.log('Z-Wave device discovery started successfully');
            onSuccess?.();
        } catch (error) {
            console.error('Error starting Z-Wave device discovery:', error);
            setError(error instanceof Error ? error.message : 'Failed to start device discovery');
            onError?.(error instanceof Error ? error : new Error('Failed to start device discovery'));
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <Paper sx={{ p: 3, maxWidth: 400, mx: 'auto', mt: 4 }}>
            <Typography variant="h6" gutterBottom>
                Add Z-Wave Device
            </Typography>
            
            <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Enter the 5-digit PIN code found on your Z-Wave device to begin the pairing process.
            </Typography>

            {error && (
                <Alert severity="error" sx={{ mb: 2 }}>
                    {error}
                </Alert>
            )}

            <Box sx={{ display: 'flex', gap: 2, alignItems: 'flex-start' }}>
                <TextField
                    label="PIN Code"
                    value={pinCode}
                    onChange={handlePinCodeChange}
                    placeholder="Enter 5-digit code"
                    inputProps={{
                        maxLength: 5,
                        pattern: '[0-9]*',
                        inputMode: 'numeric'
                    }}
                    disabled={isLoading}
                    sx={{ flex: 1 }}
                />
                
                <Button
                    variant="contained"
                    onClick={handleSubmit}
                    disabled={isLoading || pinCode.length !== 5}
                    sx={{ minWidth: 120 }}
                >
                    {isLoading ? (
                        <CircularProgress size={24} color="inherit" />
                    ) : (
                        'Start Discovery'
                    )}
                </Button>
            </Box>
        </Paper>
    );
};