import React, { useEffect, useRef, useState } from 'react';
import { Html5QrcodeScanner, Html5QrcodeScanType, Html5QrcodeSupportedFormats } from 'html5-qrcode';
import { Box, Typography, Button, Alert, IconButton, Chip } from '@mui/material';
import { QrCodeScanner, Videocam, VideocamOff, CameraAlt, PlayArrow, QrCode2 } from '@mui/icons-material';

interface SimpleQRScannerProps {
  onScanSuccess: (decodedText: string) => void;
  onScanError?: (errorMessage: string) => void;
  width?: number;
  height?: number;
}

const SimpleQRScanner: React.FC<SimpleQRScannerProps> = ({
  onScanSuccess,
  onScanError,
  width = 400,
  height = 300
}) => {
  const [isScanning, setIsScanning] = useState(false);
  const [isPaused, setIsPaused] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [lastScan, setLastScan] = useState<string>('');
  const [capturedFrame, setCapturedFrame] = useState<string | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [detectedFormat, setDetectedFormat] = useState<string>('');
  const scannerRef = useRef<Html5QrcodeScanner | null>(null);
  const videoRef = useRef<HTMLVideoElement | null>(null);
  const canvasRef = useRef<HTMLCanvasElement | null>(null);
  const scannerId = `qr-scanner-${Date.now()}`;

  const startScanner = async () => {
    try {
      setError(null);
      console.log('Starting simple QR scanner...');

      // Check camera permission
      const stream = await navigator.mediaDevices.getUserMedia({ video: true });
      stream.getTracks().forEach(track => track.stop()); // Stop the test stream

      // Clean up any existing scanner
      if (scannerRef.current) {
        try {
          await scannerRef.current.clear();
        } catch (e) {
          console.warn('Error clearing existing scanner:', e);
        }
      }

      // Create new scanner optimized for highest quality on Mac
      const scanner = new Html5QrcodeScanner(
        scannerId,
        {
          fps: 15, // Optimal FPS for Mac cameras - fast enough for responsiveness, not too high for performance
          qrbox: { width: 300, height: 300 }, // Larger detection area for better code capture
          aspectRatio: 1.0,
          disableFlip: false, // Enable flip for all orientations
          supportedScanTypes: [Html5QrcodeScanType.SCAN_TYPE_CAMERA],
          showTorchButtonIfSupported: true, // Torch for low light conditions
          showZoomSliderIfSupported: true, // Manual zoom control
          useBarCodeDetectorIfSupported: true, // Use native browser detection for best performance
          defaultZoomValueIfSupported: 2, // 2x zoom for small codes
          // Comprehensive format support for IoT devices
          formatsToSupport: [
            Html5QrcodeSupportedFormats.QR_CODE,        // Standard QR codes
            Html5QrcodeSupportedFormats.DATA_MATRIX,    // Data Matrix codes (common in IoT)
            Html5QrcodeSupportedFormats.AZTEC,          // Aztec codes
            Html5QrcodeSupportedFormats.CODE_128,       // Code 128 barcodes
            Html5QrcodeSupportedFormats.CODE_39,        // Code 39 barcodes
          ],
          // High-quality Mac camera settings
          videoConstraints: {
            facingMode: "environment", // Prefer back camera if available
            width: { ideal: 1280, min: 640 }, // High resolution for clarity
            height: { ideal: 720, min: 480 }, // HD quality
            frameRate: { ideal: 30, min: 15 } // Smooth video for better focus
          },
          // Advanced detection settings
          experimentalFeatures: {
            useBarCodeDetectorIfSupported: true
          }
        },
        false
      );

      // Set up callbacks
      const handleSuccess = (decodedText: string, decodedResult: any) => {
        console.log('Code detected:', decodedText);
        console.log('Detection result:', decodedResult);
        
        // Determine the format of the detected code
        let format = 'Unknown';
        if (decodedResult?.result?.format) {
          format = decodedResult.result.format.formatName || decodedResult.result.format;
        } else if (decodedResult?.format) {
          format = decodedResult.format.formatName || decodedResult.format;
        } else {
          // Try to guess format based on content
          if (decodedText.length === 20 && /^\d+$/.test(decodedText)) {
            format = 'Likely Z-Wave DSK';
          } else if (decodedText.length >= 32 && /^[0-9A-Fa-f]+$/.test(decodedText)) {
            format = 'Likely Zigbee Install Code';
          } else if (decodedText.startsWith('http') || decodedText.includes('://')) {
            format = 'QR Code (URL)';
          } else {
            format = 'QR Code or Matrix';
          }
        }
        
        console.log('Detected format:', format);
        setDetectedFormat(format);
        setLastScan(decodedText);
        onScanSuccess(decodedText);
        
        // Stop scanner after successful scan
        stopScanner();
      };

      const handleError = (errorMessage: string) => {
        // Only log non-routine errors
        if (!errorMessage.includes('No QR code found') && 
            !errorMessage.includes('NotFoundException') &&
            !errorMessage.includes('No barcode or QR code detected')) {
          console.warn('QR scan error:', errorMessage);
          if (onScanError) {
            onScanError(errorMessage);
          }
        }
      };

      // Render the scanner
      scanner.render(handleSuccess, handleError);
      scannerRef.current = scanner;
      setIsScanning(true);
      
      // Apply basic styling for better visibility
      setTimeout(() => {
        const scannerElement = document.getElementById(scannerId);
        
        if (scannerElement) {
          // Enhanced focus assistance with better visibility
          const scanRegion = scannerElement?.querySelector('#qr-shaded-region');
          if (scanRegion) {
            (scanRegion as HTMLElement).style.border = '4px solid #00ff00';
            (scanRegion as HTMLElement).style.boxShadow = '0 0 25px rgba(0, 255, 0, 0.7), inset 0 0 15px rgba(0, 255, 0, 0.3)';
            (scanRegion as HTMLElement).style.borderRadius = '8px';
          }
        }
      }, 1000);
      
      console.log('Simple QR scanner started successfully');

    } catch (err) {
      console.error('Failed to start QR scanner:', err);
      setError(`Failed to start camera: ${err instanceof Error ? err.message : 'Unknown error'}`);
    }
  };

  const stopScanner = async () => {
    if (scannerRef.current) {
      try {
        console.log('Stopping QR scanner...');
        await scannerRef.current.clear();
        scannerRef.current = null;
        setIsScanning(false);
        setIsPaused(false);
        setCapturedFrame(null);
        console.log('QR scanner stopped');
      } catch (err) {
        console.warn('Error stopping scanner:', err);
        setIsScanning(false);
        setIsPaused(false);
        setCapturedFrame(null);
      }
    }
  };

  const captureFrame = async () => {
    try {
      setIsProcessing(true);
      console.log('Capturing frame for QR analysis...');

      // Find the video element in the scanner
      const scannerElement = document.getElementById(scannerId);
      const videoElement = scannerElement?.querySelector('video') as HTMLVideoElement;
      
      if (!videoElement) {
        console.error('Video element not found');
        setError('Could not access camera feed');
        return;
      }

      // Create canvas to capture frame
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      
      if (!ctx) {
        console.error('Could not get canvas context');
        setError('Could not capture frame');
        return;
      }

      // Set canvas dimensions to match video
      canvas.width = videoElement.videoWidth;
      canvas.height = videoElement.videoHeight;

      // Draw current video frame to canvas
      ctx.drawImage(videoElement, 0, 0, canvas.width, canvas.height);
      
      // Convert to data URL
      const frameDataUrl = canvas.toDataURL('image/png');
      setCapturedFrame(frameDataUrl);
      setIsPaused(true);

      console.log('Frame captured successfully');

      // Try to process the captured frame with html5-qrcode
      try {
        const { Html5Qrcode } = await import('html5-qrcode');
        const html5QrCode = new Html5Qrcode(scannerId + '-capture');
        
        // Convert data URL to file
        const response = await fetch(frameDataUrl);
        const blob = await response.blob();
        const file = new File([blob], 'captured-frame.png', { type: 'image/png' });

        // Try to scan the captured image
        const result = await html5QrCode.scanFile(file, true);
        console.log('QR Code found in captured frame:', result);
        
        setLastScan(result);
        onScanSuccess(result);
        stopScanner();
        
      } catch (scanError) {
        console.log('No QR code found in captured frame, continuing live scan...');
        // If no QR code found in captured frame, resume live scanning after a delay
        setTimeout(() => {
          setIsPaused(false);
          setCapturedFrame(null);
        }, 10000); // Give user 10 seconds to see the captured frame
      }

    } catch (error) {
      console.error('Error capturing frame:', error);
      setError('Failed to capture frame');
      setIsPaused(false);
      setCapturedFrame(null);
    } finally {
      setIsProcessing(false);
    }
  };

  const resumeScanning = () => {
    setIsPaused(false);
    setCapturedFrame(null);
    console.log('Resumed live scanning');
  };

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      if (scannerRef.current) {
        scannerRef.current.clear().catch(console.warn);
      }
    };
  }, []);

  return (
    <Box sx={{ width: '100%', maxWidth: width, margin: '0 auto' }}>
      {/* Error Alert */}
      {error && (
        <Alert severity="error" sx={{ mb: 2 }}>
          <Typography variant="body2">{error}</Typography>
        </Alert>
      )}

      {/* Success Alert */}
      {lastScan && (
        <Alert severity="success" sx={{ mb: 2 }}>
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
            <QrCode2 fontSize="small" />
            <Typography variant="body2" sx={{ fontWeight: 'bold' }}>
              Code Detected!
            </Typography>
            {detectedFormat && (
              <Chip 
                label={detectedFormat} 
                size="small" 
                color="success" 
                variant="outlined"
              />
            )}
          </Box>
          <Typography variant="body2">
            <strong>Content:</strong> {lastScan.length > 50 ? `${lastScan.substring(0, 50)}...` : lastScan}
          </Typography>
        </Alert>
      )}

      {/* Scanner Container */}
      <Box
        sx={{
          position: 'relative',
          width: '100%',
          minHeight: height,
          border: '2px dashed',
          borderColor: isScanning ? 'primary.main' : 'grey.300',
          borderRadius: 2,
          overflow: 'hidden',
          backgroundColor: 'background.paper',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center'
        }}
      >
        {/* Scanner Placeholder */}
        {!isScanning && (
          <Box sx={{ textAlign: 'center', p: 3 }}>
            <QrCodeScanner sx={{ fontSize: 48, color: 'grey.400', mb: 2 }} />
            <Typography variant="body2" color="text.secondary" sx={{ mb: 1 }}>
              Click "Start Scanner" to begin
            </Typography>
            <Typography variant="caption" color="text.secondary">
              Supports QR codes, Data Matrix, Aztec, and various barcodes
            </Typography>
          </Box>
        )}

        {/* Captured Frame Overlay */}
        {capturedFrame && isPaused && (
          <Box
            sx={{
              position: 'absolute',
              top: 0,
              left: 0,
              width: '100%',
              height: '100%',
              zIndex: 10,
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              bgcolor: 'rgba(0, 0, 0, 0.8)'
            }}
          >
            <Box sx={{ textAlign: 'center', p: 2 }}>
              <img
                src={capturedFrame}
                alt="Captured frame"
                style={{
                  maxWidth: '100%',
                  maxHeight: '200px',
                  borderRadius: '8px',
                  border: '2px solid #fff'
                }}
              />
              <Typography variant="body2" color="white" sx={{ mt: 1 }}>
                {isProcessing ? 'Processing captured frame...' : 'Frame captured - analyzing for QR code'}
              </Typography>
            </Box>
          </Box>
        )}

        {/* Scanner Element */}
        <div 
          id={scannerId} 
          style={{ 
            width: '100%',
            minHeight: isScanning ? height : 0,
            display: isScanning ? 'block' : 'none',
            opacity: isPaused ? 0.3 : 1,
            transition: 'opacity 0.3s ease'
          }} 
        />

        {/* Capture Button Overlay */}
        {isScanning && !isPaused && (
          <IconButton
            onClick={captureFrame}
            disabled={isProcessing}
            sx={{
              position: 'absolute',
              bottom: 16,
              right: 16,
              bgcolor: 'primary.main',
              color: 'white',
              '&:hover': {
                bgcolor: 'primary.dark'
              },
              zIndex: 5
            }}
          >
            <CameraAlt />
          </IconButton>
        )}

        {/* Resume Button */}
        {isPaused && !isProcessing && (
          <IconButton
            onClick={resumeScanning}
            sx={{
              position: 'absolute',
              bottom: 16,
              right: 16,
              bgcolor: 'success.main',
              color: 'white',
              '&:hover': {
                bgcolor: 'success.dark'
              },
              zIndex: 15
            }}
          >
            <PlayArrow />
          </IconButton>
        )}
      </Box>

      {/* Control Buttons */}
      <Box sx={{ mt: 2, textAlign: 'center', display: 'flex', gap: 2, justifyContent: 'center' }}>
        {isScanning ? (
          <>
            <Button
              variant="outlined"
              color="error"
              startIcon={<VideocamOff />}
              onClick={stopScanner}
            >
              Stop Scanner
            </Button>
            {!isPaused && (
              <Button
                variant="contained"
                color="primary"
                startIcon={<CameraAlt />}
                onClick={captureFrame}
                disabled={isProcessing}
              >
                {isProcessing ? 'Processing...' : 'Capture Frame'}
              </Button>
            )}
            {isPaused && (
              <Button
                variant="contained"
                color="success"
                startIcon={<PlayArrow />}
                onClick={resumeScanning}
              >
                Resume Live Scan
              </Button>
            )}
          </>
        ) : (
          <Button
            variant="contained"
            color="primary"
            startIcon={<Videocam />}
            onClick={startScanner}
          >
            Start Scanner
          </Button>
        )}
      </Box>

      {/* Scanner Status and Supported Formats */}
      <Box sx={{ mt: 1, textAlign: 'center' }}>
        <Typography variant="caption" color="text.secondary" sx={{ display: 'block', mb: 0.5 }}>
          Status: {
            isProcessing ? 'Processing captured frame...' :
            isPaused ? 'Frame captured - analyzing for codes' :
            isScanning ? 'Live scanning - optimized for Data Matrix codes' : 
            'Ready to scan'
          }
        </Typography>
        {!isScanning && (
          <Typography variant="caption" color="text.secondary" sx={{ fontSize: '0.7rem' }}>
            Supported: QR • Data Matrix • Aztec • Code 128/93/39 • EAN • PDF417 • ITF
          </Typography>
        )}
      </Box>
    </Box>
  );
};

export default SimpleQRScanner;
