// src/components/DeveloperConfig.tsx

import React, { useState, useEffect } from 'react';
import {
    Box,
    Card,
    CardContent,
    TextField,
    Button,
    Typography,
    Alert,
    Chip,
    Grid,
    FormControl,
    InputLabel,
    Select,
    MenuItem,
    Divider,
    CircularProgress
} from '@mui/material';
import { Settings, CloudQueue, CheckCircle, Error } from '@mui/icons-material';
import CustomEndpointService, { CustomEndpointConfig } from '../services/customEndpointService';

interface DeveloperConfigProps {
    onConfigChange?: (config: CustomEndpointConfig) => void;
}

const AWS_REGIONS = [
    'us-east-1',
    'us-east-2', 
    'us-west-1',
    'us-west-2',
    'eu-west-1',
    'eu-west-2',
    'eu-central-1',
    'ap-southeast-1',
    'ap-southeast-2',
    'ap-northeast-1'
];

const DeveloperConfig: React.FC<DeveloperConfigProps> = ({ onConfigChange }) => {
    const [config, setConfig] = useState<CustomEndpointConfig>({
        region: process.env.REACT_APP_AWS_REGION || 'us-east-1',
        endpoint: process.env.REACT_APP_IOT_ENDPOINT || '',
        credentials: {
            accessKeyId: process.env.REACT_APP_AWS_ACCESS_KEY_ID || '',
            secretAccessKey: process.env.REACT_APP_AWS_SECRET_ACCESS_KEY || '',
            sessionToken: process.env.REACT_APP_AWS_SESSION_TOKEN || ''
        }
    });

    const [customEndpointId, setCustomEndpointId] = useState<string>('');
    const [testResult, setTestResult] = useState<{ success: boolean; error?: string } | null>(null);
    const [endpointResult, setEndpointResult] = useState<any>(null);
    const [isLoading, setIsLoading] = useState(false);
    const [customEndpointService, setCustomEndpointService] = useState<CustomEndpointService | null>(null);

    // Initialize service when config changes
    useEffect(() => {
        if (config.credentials.accessKeyId && config.credentials.secretAccessKey) {
            const service = new CustomEndpointService(config);
            setCustomEndpointService(service);
            onConfigChange?.(config);
        }
    }, [config, onConfigChange]);

    const handleConfigChange = (field: keyof CustomEndpointConfig | string, value: string) => {
        if (field === 'region' || field === 'endpoint') {
            setConfig(prev => ({
                ...prev,
                [field]: value
            }));
        } else if (field.startsWith('credentials.')) {
            const credField = field.split('.')[1] as keyof typeof config.credentials;
            setConfig(prev => ({
                ...prev,
                credentials: {
                    ...prev.credentials,
                    [credField]: value
                }
            }));
        }
    };

    const testConnection = async () => {
        if (!customEndpointService) return;
        
        setIsLoading(true);
        setTestResult(null);
        
        try {
            const result = await customEndpointService.testConnection();
            setTestResult(result);
        } catch (error) {
            setTestResult({
                success: false,
                error: 'Connection test failed'
            });
        } finally {
            setIsLoading(false);
        }
    };

    const getCustomEndpoint = async () => {
        if (!customEndpointService || !customEndpointId.trim()) {
            alert('Please enter a Custom Endpoint ID');
            return;
        }
        
        setIsLoading(true);
        setEndpointResult(null);
        
        try {
            const result = await customEndpointService.getCustomEndpoint(customEndpointId.trim());
            setEndpointResult(result);
        } catch (error) {
            setEndpointResult({
                success: false,
                error: 'Failed to get custom endpoint'
            });
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <Box sx={{ p: 2 }}>
            <Card>
                <CardContent>
                    <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                        <Settings sx={{ mr: 1 }} />
                        <Typography variant="h6">Developer Configuration</Typography>
                        <Chip 
                            label="DEV MODE" 
                            color="warning" 
                            size="small" 
                            sx={{ ml: 2 }} 
                        />
                    </Box>

                    <Grid container spacing={3}>
                        {/* AWS Configuration */}
                        <Grid size={{ xs: 12, md: 6 }}>
                            <Typography variant="subtitle1" gutterBottom>
                                AWS Configuration
                            </Typography>
                            
                            <FormControl fullWidth margin="normal">
                                <InputLabel>Region</InputLabel>
                                <Select
                                    value={config.region}
                                    label="Region"
                                    onChange={(e) => handleConfigChange('region', e.target.value)}
                                >
                                    {AWS_REGIONS.map(region => (
                                        <MenuItem key={region} value={region}>
                                            {region}
                                        </MenuItem>
                                    ))}
                                </Select>
                            </FormControl>

                            <TextField
                                fullWidth
                                label="Custom API Endpoint"
                                value={config.endpoint}
                                onChange={(e) => handleConfigChange('endpoint', e.target.value)}
                                margin="normal"
                                placeholder="https://api.gamma.us-east-1.iotmanagedintegrations.iot.aws.dev"
                                helperText="Leave empty to use default AWS endpoint"
                            />

                            <TextField
                                fullWidth
                                label="Access Key ID"
                                value={config.credentials.accessKeyId}
                                onChange={(e) => handleConfigChange('credentials.accessKeyId', e.target.value)}
                                margin="normal"
                                type="password"
                            />

                            <TextField
                                fullWidth
                                label="Secret Access Key"
                                value={config.credentials.secretAccessKey}
                                onChange={(e) => handleConfigChange('credentials.secretAccessKey', e.target.value)}
                                margin="normal"
                                type="password"
                            />

                            <TextField
                                fullWidth
                                label="Session Token (Optional)"
                                value={config.credentials.sessionToken}
                                onChange={(e) => handleConfigChange('credentials.sessionToken', e.target.value)}
                                margin="normal"
                                type="password"
                            />

                            <Box sx={{ mt: 2 }}>
                                <Button
                                    variant="outlined"
                                    onClick={testConnection}
                                    disabled={isLoading || !config.credentials.accessKeyId || !config.credentials.secretAccessKey}
                                    startIcon={isLoading ? <CircularProgress size={20} /> : <CloudQueue />}
                                >
                                    Test Connection
                                </Button>
                            </Box>

                            {testResult && (
                                <Alert 
                                    severity={testResult.success ? 'success' : 'error'} 
                                    sx={{ mt: 2 }}
                                    icon={testResult.success ? <CheckCircle /> : <Error />}
                                >
                                    {testResult.success 
                                        ? 'Connection successful!' 
                                        : `Connection failed: ${testResult.error}`
                                    }
                                </Alert>
                            )}
                        </Grid>

                        {/* Custom Endpoint Testing */}
                        <Grid size={{ xs: 12, md: 6 }}>
                            <Typography variant="subtitle1" gutterBottom>
                                Custom Endpoint Testing
                            </Typography>
                            
                            <TextField
                                fullWidth
                                label="Custom Endpoint ID"
                                value={customEndpointId}
                                onChange={(e) => setCustomEndpointId(e.target.value)}
                                margin="normal"
                                placeholder="Enter custom endpoint identifier"
                                helperText="ID of the custom endpoint to retrieve"
                            />

                            <Box sx={{ mt: 2 }}>
                                <Button
                                    variant="contained"
                                    onClick={getCustomEndpoint}
                                    disabled={isLoading || !customEndpointId.trim() || !customEndpointService}
                                    startIcon={isLoading ? <CircularProgress size={20} /> : <Settings />}
                                >
                                    Get Custom Endpoint
                                </Button>
                            </Box>

                            {endpointResult && (
                                <Box sx={{ mt: 2 }}>
                                    {endpointResult.success ? (
                                        <Alert severity="success" sx={{ mb: 2 }}>
                                            Custom endpoint retrieved successfully!
                                        </Alert>
                                    ) : (
                                        <Alert severity="error" sx={{ mb: 2 }}>
                                            {endpointResult.error}
                                        </Alert>
                                    )}

                                    {endpointResult.customEndpoint && (
                                        <Card variant="outlined">
                                            <CardContent>
                                                <Typography variant="subtitle2" gutterBottom>
                                                    Custom Endpoint Details:
                                                </Typography>
                                                <pre style={{ 
                                                    fontSize: '12px', 
                                                    overflow: 'auto',
                                                    backgroundColor: '#f5f5f5',
                                                    padding: '8px',
                                                    borderRadius: '4px'
                                                }}>
                                                    {JSON.stringify(endpointResult.customEndpoint, null, 2)}
                                                </pre>
                                            </CardContent>
                                        </Card>
                                    )}
                                </Box>
                            )}
                        </Grid>
                    </Grid>

                    <Divider sx={{ my: 3 }} />

                    <Typography variant="body2" color="text.secondary">
                        <strong>Current Configuration:</strong><br />
                        Region: {config.region}<br />
                        Endpoint: {config.endpoint || 'Default AWS endpoint'}<br />
                        Credentials: {config.credentials.accessKeyId ? 'Configured' : 'Not configured'}
                    </Typography>
                </CardContent>
            </Card>
        </Box>
    );
};

export default DeveloperConfig;
