import React from 'react';
import {
    Box,
    Typography,
    Paper,
    Stack,
    Chip,
    Card,
    CardContent
} from '@mui/material';
import * as Icons from '@mui/icons-material';
import { Capability } from '../types/deviceTypes';
import { ClusterImplementation, ClusterProps } from '../types/clusterTypes';

// Helper function to format temperature value
const formatTemperature = (value: any): string => {
    if (!value || typeof value !== 'object') return 'N/A';
    
    const { temperature, scale } = value;
    if (temperature === undefined || temperature === null) return 'N/A';
    
    return `${temperature}°${scale || 'C'}`;
};

// Helper function to get temperature color based on value
const getTemperatureColor = (temp: number): string => {
    if (temp < 10) return '#2196F3'; // Blue for cold
    if (temp < 20) return '#4CAF50'; // Green for cool
    if (temp < 25) return '#FF9800'; // Orange for warm
    return '#F44336'; // Red for hot
};

// Details view component
const TemperatureMeasurementDetails: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const measuredValueProperty = capability.properties.find(p => p.name === 'MeasuredValue');
    const minValueProperty = capability.properties.find(p => p.name === 'MinMeasuredValue');
    const maxValueProperty = capability.properties.find(p => p.name === 'MaxMeasuredValue');

    const currentTemp = measuredValueProperty?.value?.propertyValue;
    const tempValue = currentTemp?.temperature || 0;
    const tempScale = currentTemp?.scale || 'C';
    const tempColor = getTemperatureColor(tempValue);

    return (
        <Box>
            <Box sx={{ 
                display: 'flex', 
                alignItems: 'center', 
                gap: 3,
                mb: 2,
                p: 3,
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                position: 'relative',
                transition: 'all 0.3s ease',
                '&:hover': {
                    boxShadow: 2
                }
            }}>
                <Box sx={{
                    position: 'relative',
                    width: 64,
                    height: 64,
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    borderRadius: '50%',
                    bgcolor: `${tempColor}20`,
                    border: `2px solid ${tempColor}`,
                    transition: 'all 0.3s ease'
                }}>
                    <Icons.Thermostat sx={{ 
                        fontSize: 40, 
                        color: tempColor,
                        transition: 'all 0.3s ease'
                    }} />
                </Box>

                <Box sx={{ flexGrow: 1 }}>
                    <Typography variant="h4" sx={{ 
                        color: tempColor,
                        fontWeight: 'bold',
                        mb: 1
                    }}>
                        {formatTemperature(currentTemp)}
                    </Typography>
                    <Typography variant="h6" color="text.secondary">
                        {capability.name}
                    </Typography>
                    {measuredValueProperty?.value?.lastChangedAt && (
                        <Typography variant="caption" color="text.secondary">
                            Last updated: {new Date(measuredValueProperty.value.lastChangedAt).toLocaleString()}
                        </Typography>
                    )}
                </Box>
            </Box>

            <Stack spacing={2} sx={{ mt: 3 }}>
                {minValueProperty && (
                    <Card variant="outlined">
                        <CardContent sx={{ py: 2 }}>
                            <Typography variant="body2" color="text.secondary" gutterBottom>
                                Minimum Measured Value
                            </Typography>
                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                <Typography variant="h6">
                                    {formatTemperature(minValueProperty.value?.propertyValue)}
                                </Typography>
                                {minValueProperty.value?.lastChangedAt && (
                                    <Typography variant="caption" color="text.secondary">
                                        (Last updated: {new Date(minValueProperty.value.lastChangedAt).toLocaleString()})
                                    </Typography>
                                )}
                            </Box>
                        </CardContent>
                    </Card>
                )}

                {maxValueProperty && (
                    <Card variant="outlined">
                        <CardContent sx={{ py: 2 }}>
                            <Typography variant="body2" color="text.secondary" gutterBottom>
                                Maximum Measured Value
                            </Typography>
                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                <Typography variant="h6">
                                    {formatTemperature(maxValueProperty.value?.propertyValue)}
                                </Typography>
                                {maxValueProperty.value?.lastChangedAt && (
                                    <Typography variant="caption" color="text.secondary">
                                        (Last updated: {new Date(maxValueProperty.value.lastChangedAt).toLocaleString()})
                                    </Typography>
                                )}
                            </Box>
                        </CardContent>
                    </Card>
                )}

                {capability.actions && capability.actions.length > 0 && (
                    <Box>
                        <Typography variant="body2" color="text.secondary">
                            Available Actions:
                        </Typography>
                        <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1, mt: 0.5 }}>
                            {capability.actions.map((action) => (
                                <Chip
                                    key={action}
                                    label={action}
                                    size="small"
                                    variant="outlined"
                                />
                            ))}
                        </Box>
                    </Box>
                )}
            </Stack>
        </Box>
    );
};

// Widget for dashboard cards
export const TemperatureMeasurementWidget: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const measuredValueProperty = capability.properties.find(p => p.name === 'MeasuredValue');
    const currentTemp = measuredValueProperty?.value?.propertyValue;
    const tempValue = currentTemp?.temperature || 0;
    const tempColor = getTemperatureColor(tempValue);

    return (
        <Box sx={{ 
            p: 2, 
            borderRadius: 2,
            bgcolor: 'background.paper',
            position: 'relative',
            minHeight: 100,
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
            gap: 2
        }}>
            <Box sx={{
                position: 'relative',
                width: 48,
                height: 48,
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                borderRadius: '50%',
                bgcolor: `${tempColor}20`,
                border: `2px solid ${tempColor}`,
                transition: 'all 0.3s ease'
            }}>
                <Icons.Thermostat sx={{ 
                    fontSize: 32, 
                    color: tempColor,
                    transition: 'all 0.3s ease'
                }} />
            </Box>
            
            <Box sx={{ textAlign: 'center' }}>
                <Typography variant="h6" sx={{ 
                    color: tempColor,
                    fontWeight: 'bold'
                }}>
                    {formatTemperature(currentTemp)}
                </Typography>
                <Typography variant="body2" color="text.secondary">
                    Temperature
                </Typography>
            </Box>
        </Box>
    );
};

// Cluster implementation
const TemperatureMeasurementCluster: ClusterImplementation = Object.assign(
    // The default component render
    ({ capability, deviceId }: ClusterProps) => {
        return <TemperatureMeasurementDetails capability={capability} deviceId={deviceId} />;
    },
    // The interface methods
    {
        renderDetails: ({ capability, deviceId }: ClusterProps) => {
            return <TemperatureMeasurementDetails capability={capability} deviceId={deviceId} />;
        },
        renderWidget: ({ capability, deviceId }: ClusterProps) => {
            return <TemperatureMeasurementWidget capability={capability} deviceId={deviceId} />;
        }
    }
);

export default TemperatureMeasurementCluster;
