import React, { useState, useEffect } from 'react';
import {
    Box,
    Typography,
    Switch,
    FormControlLabel,
    Chip,
    Stack,
    IconButton,
    CircularProgress,
    styled,
    SwitchProps,
    Paper,
} from '@mui/material';
import * as Icons from '@mui/icons-material';
import { Capability, PropertyValue } from '../types/deviceTypes';
import { deviceService } from '../services/deviceService';
import { ClusterImplementation, ClusterProps } from '../types/clusterTypes';
import { useDispatch } from 'react-redux';
import { updateDeviceState } from '../store/deviceSlice';
import { deviceStatePollingService } from '../services/deviceStatePollingService';
import { debugLogger } from '../utils/debugLogger';

// Custom iOS style switch
const IOSSwitch = ({ checked, onChange, disabled }: { checked: boolean; onChange: () => void; disabled?: boolean }) => (
    <Switch
        checked={checked}
        onChange={onChange}
        disabled={disabled}
        sx={{
            width: 42,
            height: 26,
            padding: 0,
            '& .MuiSwitch-switchBase': {
                padding: 0,
                margin: '2px',
                transitionDuration: '300ms',
                '&.Mui-checked': {
                    transform: 'translateX(16px)',
                    color: '#fff',
                    '& + .MuiSwitch-track': {
                        backgroundColor: '#65C466',
                        opacity: 1,
                        border: 0,
                    },
                    '&.Mui-disabled + .MuiSwitch-track': {
                        opacity: 0.5,
                    },
                },
                '&.Mui-disabled .MuiSwitch-thumb': {
                    opacity: 0.5,
                },
            },
            '& .MuiSwitch-thumb': {
                boxSizing: 'border-box',
                width: 22,
                height: 22,
                margin: 0,
            },
            '& .MuiSwitch-track': {
                borderRadius: 13,
                backgroundColor: '#E9E9EA',
                opacity: 1,
                transition: 'background-color 500ms',
                '.Mui-disabled + &': {
                    opacity: 0.5,
                },
            },
        }}
    />
);

// Helper function to format property values
const formatValue = (value: any): string => {
    if (value === undefined || value === null) return 'N/A';
    if (typeof value === 'boolean') return value ? 'Yes' : 'No';
    if (typeof value === 'number') return value.toString();
    if (typeof value === 'string') return value;
    return JSON.stringify(value);
};

// Common styles for the icon container
const getIconContainerStyles = (isOn: boolean, isDetails: boolean = false) => ({
    position: 'relative',
    width: isDetails ? 64 : 48,
    height: isDetails ? 64 : 48,
    display: 'flex',
    alignItems: 'center',
    justifyContent: 'center',
    borderRadius: '50%',
    bgcolor: isOn ? 'warning.light' : 'grey.200',
    transition: 'all 0.3s ease',
    boxShadow: isOn ? '0 0 20px rgba(255, 167, 38, 0.5)' : 'none',
    '&:hover': {
        bgcolor: isOn ? 'warning.main' : 'grey.300',
        transform: 'scale(1.05)'
    }
});

// Common styles for the icon
const getIconStyles = (isOn: boolean, isDetails: boolean = false) => ({
    fontSize: isDetails ? 40 : 32,
    color: isOn ? '#FFD700' : 'grey.600',
    filter: isOn ? 'drop-shadow(0 0 10px rgba(255, 215, 0, 0.7))' : 'none',
    animation: isOn ? 'pulse 2s infinite' : 'none',
    transition: 'all 0.3s ease'
});

// Details view component
const OnOffDetails: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const dispatch = useDispatch();
    const [isLoading, setIsLoading] = useState(false);
    const [isOn, setIsOn] = useState(() => {
        const initialValue = capability.properties[0]?.value?.propertyValue === true;
        debugLogger.log(`OnOffDetails INITIALIZATION - Device: ${deviceId}, Property:`, capability.properties[0], 'InitialValue:', initialValue);
        return initialValue;
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0); // Track when the user last clicked
    const [lastCommandTimestamp, setLastCommandTimestamp] = useState<number>(0); // Track when we sent the last command
    const [readStateInitialized, setReadStateInitialized] = useState(false); // Track if ReadState was sent
    const onTimeProperty = capability.properties.find(p => p.name === 'OnTime');
    const offWaitTimeProperty = capability.properties.find(p => p.name === 'OffWaitTime');

    // Send ReadState command on initialization for light devices - DISABLED FOR TESTING
    useEffect(() => {
        // TEMPORARILY DISABLED: Testing if ReadState commands are causing CDMB crashes
        debugLogger.log(`💡 OnOffDetails - ReadState DISABLED for testing - device ${deviceId}`);
        setReadStateInitialized(true); // Mark as initialized without sending ReadState
        
        // if (!readStateInitialized && deviceId && capability) {
        //     console.log(`OnOffDetails - Sending ReadState for light device ${deviceId} on initialization`);
        //     
        //     // Send ReadState command to get fresh state
        //     deviceService.sendReadStateForLightDevice(deviceId)
        //         .then(() => {
        //             console.log(`ReadState sent successfully for light device ${deviceId}`);
        //             setReadStateInitialized(true);
        //         })
        //         .catch((error) => {
        //             console.error(`Failed to send ReadState for light device ${deviceId}:`, error);
        //             // Still mark as initialized to prevent retries
        //             setReadStateInitialized(true);
        //         });
        // }
    }, [deviceId, capability, readStateInitialized]);

    // Update local state when capability changes, but respect optimistic updates
    useEffect(() => {
        const now = Date.now();
        const timeSinceUserAction = now - lastUserAction;
        const newValue = capability.properties[0]?.value?.propertyValue === true;
        const lastChangedAt = capability.properties[0]?.value?.lastChangedAt;
        
        debugLogger.log('OnOffDetails - Capability updated:', {
            deviceId,
            timeSinceUserAction,
            currentIsOn: isOn,
            newValue,
            lastChangedAt,
            lastCommandTimestamp,
            timestamp: new Date().toISOString()
        });
        
        // Check if we should ignore this update based on timestamp comparison
        // Use the centralized command tracking from deviceStatePollingService
        if (lastChangedAt) {
            const deviceStateTimestamp = new Date(lastChangedAt).getTime();
            
            // Get the most recent command timestamp from the polling service (not just our local one)
            const mostRecentCommandTime = deviceStatePollingService.getLastCommandTimestamp(deviceId) || lastCommandTimestamp;
            
            if (mostRecentCommandTime > 0) {
                const timeDifference = deviceStateTimestamp - mostRecentCommandTime;
                const absTimeDifference = Math.abs(timeDifference);
                
                // Clock sync detection: Log potential issues with timezone analysis
                const now = Date.now();
                const localTime = new Date(now);
                const deviceTime = new Date(lastChangedAt);
                const commandTime = new Date(mostRecentCommandTime);
                
                const clockSyncInfo = {
                    deviceId,
                    // Timestamps in milliseconds for precise comparison
                    nowMs: now,
                    commandMs: mostRecentCommandTime,
                    deviceStateMs: deviceStateTimestamp,
                    // Human readable times
                    nowISO: localTime.toISOString(),
                    commandISO: commandTime.toISOString(),
                    deviceStateISO: lastChangedAt,
                    // Local timezone info
                    localTimezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                    localOffset: localTime.getTimezoneOffset(),
                    // Time differences
                    timeDifferenceMs: timeDifference,
                    timeDifferenceSeconds: Math.round(timeDifference / 1000),
                    timeDifferenceMinutes: Math.round(timeDifference / 60000),
                    // Analysis
                    deviceVsNow: Math.round((deviceStateTimestamp - now) / 1000),
                    commandVsNow: Math.round((mostRecentCommandTime - now) / 1000),
                    currentIsOn: isOn,
                    newValue
                };
                
                // Warn about potential clock sync issues
                if (absTimeDifference > 60000) { // More than 1 minute difference
                    debugLogger.warn('POTENTIAL CLOCK SYNC ISSUE - Large timestamp difference detected:', {
                        ...clockSyncInfo,
                        severity: absTimeDifference > 300000 ? 'CRITICAL (>5min)' : 'WARNING (>1min)',
                        recommendation: 'Check if device/server clocks are synchronized'
                    });
                }
                
                // If the MOST RECENT command (from any component) was sent AFTER the device state timestamp, ignore this update
                if (mostRecentCommandTime > deviceStateTimestamp) {
                    debugLogger.log('OnOffDetails - Ignoring stale device state (older than MOST RECENT command):', clockSyncInfo);
                    return;
                }
                
                // If device state is newer than the most recent command, accept it
                debugLogger.log('OnOffDetails - Accepting newer device state (newer than MOST RECENT command):', clockSyncInfo);
            }
        }
        
        // Hybrid approach: timestamp comparison + grace period fallback for clock sync issues
        const mostRecentCommandTime = deviceStatePollingService.getLastCommandTimestamp(deviceId) || lastCommandTimestamp;
        
        if (mostRecentCommandTime === 0) {
            // Initial load - no commands sent yet, trust the device state
            setIsOn(newValue);
            debugLogger.log('OnOffDetails - Initial load, accepting device state:', {
                deviceId,
                newValue,
                timestamp: new Date().toISOString()
            });
        } else if (lastChangedAt) {
            const deviceStateTimestamp = new Date(lastChangedAt).getTime();
            const timeSinceCommand = Date.now() - mostRecentCommandTime;
            
            // Primary check: timestamp comparison + minimum threshold
            // Only accept device state if it's newer than command AND at least 5 seconds have passed
            if (deviceStateTimestamp > mostRecentCommandTime && timeSinceCommand > 5000) {
                // Device state is newer than our most recent command AND enough time has passed
                setIsOn(newValue);
                console.log('OnOffDetails - Device state newer than command + threshold met, accepting:', {
                    deviceId,
                    deviceStateTimestamp: lastChangedAt,
                    mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    newValue,
                    method: 'timestamp-comparison-with-threshold'
                });
            }
            // Check if timestamp is newer but we're still within the 5-second threshold
            else if (deviceStateTimestamp > mostRecentCommandTime && timeSinceCommand <= 5000) {
                console.log('OnOffDetails - Device state newer but within 5s threshold, keeping optimistic state:', {
                    deviceId,
                    deviceStateTimestamp: lastChangedAt,
                    mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    keepingState: isOn,
                    deviceWants: newValue,
                    method: 'threshold-protection'
                });
            }
            // Fallback check: grace period (works when clocks aren't synced)
            else if (timeSinceCommand > 15000) {
                // It's been more than 8 seconds since command - accept device state as final truth
                setIsOn(newValue);
                console.log('OnOffDetails - Grace period expired, accepting device state:', {
                    deviceId,
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    deviceStateTimestamp: lastChangedAt,
                    mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                    newValue,
                    method: 'grace-period-fallback'
                });
            } 
            else {
                console.log('OnOffDetails - Keeping optimistic state (timestamp older + within grace period):', {
                    deviceId,
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    deviceStateTimestamp: lastChangedAt,
                    mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                    keepingState: isOn,
                    deviceWants: newValue,
                    method: 'optimistic-hold'
                });
            }
        } else {
            // No timestamp available - use grace period only
            const timeSinceCommand = Date.now() - mostRecentCommandTime;
            if (timeSinceCommand > 8000) {
                setIsOn(newValue);
                console.log('OnOffDetails - No timestamp, grace period expired, accepting device state:', {
                    deviceId,
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    newValue,
                    method: 'grace-period-only'
                });
            } else {
                console.log('OnOffDetails - No timestamp, within grace period, keeping optimistic state:', {
                    deviceId,
                    timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                    keepingState: isOn,
                    deviceWants: newValue,
                    method: 'optimistic-hold-no-timestamp'
                });
            }
        }
    }, [capability.properties[0]?.value?.propertyValue, capability.properties[0]?.value?.lastChangedAt, deviceId, lastUserAction, lastCommandTimestamp]);

    const handleToggle = async () => {
        if (isLoading) return;
        
        // Store original state for potential revert
        const originalState = isOn;
        
        try {
            // Record user action timestamp to prevent polling interference
            const commandTimestamp = Date.now();
            setLastUserAction(commandTimestamp);
            setLastCommandTimestamp(commandTimestamp);
            
            // Record toggle command for sequence-based state tracking
            deviceStatePollingService.recordToggle(deviceId, originalState, !originalState, commandTimestamp);
            
            // Optimistic Update: Immediately flip the UI for responsive feel
            setIsOn(!isOn);
            console.log('OnOffDetails - Optimistic UI update:', {
                deviceId,
                from: originalState,
                to: !originalState,
                timestamp: new Date().toISOString()
            });
            
            setIsLoading(true);
            
            // Get the device details to get the endpoint ID from capability report
            const device = await deviceService.getDevice(deviceId);
            const endpointId = device.CapabilityReport.endpoints[0]?.id;
            
            if (!endpointId) {
                console.error('No endpoint ID found in capability report for device:', deviceId);
                // Revert optimistic update on error
                setIsOn(originalState);
                return;
            }

            console.log('Sending command with endpoint ID:', endpointId);

            // Send the appropriate command based on ORIGINAL state (before optimistic update)
            const command = originalState ? 'Off' : 'On';
            
            // Log command timing for comparison with GetState responses
            console.log(`COMMAND_SENT: Device ${deviceId}, Command ${command}, Timestamp ${commandTimestamp}, ISO ${new Date(commandTimestamp).toISOString()}`);
            
            await deviceService.executeAction(deviceId, capability.id, command, endpointId);

            console.log('OnOffDetails - Command sent successfully:', {
                deviceId,
                command,
                endpointId,
                timestamp: new Date().toISOString()
            });

            // Remove manual state fetching - let background polling handle validation
            // The DeviceStatePollingService will automatically correct any inconsistencies

        } catch (error) {
            console.error('Error toggling device:', error);
            // Revert optimistic update on error, but extend grace period
            // to prevent immediate polling interference
            setIsOn(originalState);
            setLastUserAction(Date.now()); // Reset grace period to prevent polling flicker
            console.log('OnOffDetails - Reverted UI due to error:', {
                deviceId,
                revertedTo: originalState,
                error: error instanceof Error ? error.message : 'Unknown error',
                timestamp: new Date().toISOString()
            });
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <Box>
            <Box sx={{ 
                display: 'flex', 
                alignItems: 'center', 
                gap: 3,
                mb: 2,
                p: 3,
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                position: 'relative',
                transition: 'all 0.3s ease',
                '&:hover': {
                    boxShadow: 2
                }
            }}>
                <Box sx={getIconContainerStyles(isOn, true)}>
                    {isOn ? (
                        <Icons.Lightbulb sx={getIconStyles(isOn, true)} />
                    ) : (
                        <Icons.LightbulbOutlined sx={getIconStyles(isOn, true)} />
                    )}
                    {isOn && (
                        <Box sx={{
                            position: 'absolute',
                            top: '50%',
                            left: '50%',
                            transform: 'translate(-50%, -50%)',
                            width: 56,
                            height: 56,
                            borderRadius: '50%',
                            bgcolor: 'warning.main',
                            opacity: 0.2,
                            animation: 'pulse 2s infinite'
                        }} />
                    )}
                </Box>

                <Box sx={{ flexGrow: 1 }}>
                    <FormControlLabel
                        control={
                            <IOSSwitch
                                checked={isOn}
                                onChange={handleToggle}
                                disabled={isLoading}
                            />
                        }
                        label={
                            <Typography 
                                variant="h6"
                                color={isOn ? 'warning.main' : 'text.secondary'}
                                sx={{ ml: 2 }}
                            >
                                {capability.name}
                            </Typography>
                        }
                    />
                    {isLoading && <Chip label="Updating..." size="small" sx={{ ml: 2 }} />}
                </Box>
            </Box>

            <Stack spacing={2} sx={{ mt: 3 }}>
                {onTimeProperty && (
                    <Box>
                        <Typography variant="body2" color="text.secondary">
                            On Time
                        </Typography>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <Typography variant="body1">
                                {formatValue(onTimeProperty.value?.propertyValue)} seconds
                            </Typography>
                            {onTimeProperty.value?.lastChangedAt && (
                                <Typography variant="caption" color="text.secondary">
                                    (Last updated: {new Date(onTimeProperty.value.lastChangedAt).toLocaleString()})
                                </Typography>
                            )}
                        </Box>
                    </Box>
                )}

                {offWaitTimeProperty && (
                    <Box>
                        <Typography variant="body2" color="text.secondary">
                            Off Wait Time
                        </Typography>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <Typography variant="body1">
                                {formatValue(offWaitTimeProperty.value?.propertyValue)} seconds
                            </Typography>
                            {offWaitTimeProperty.value?.lastChangedAt && (
                                <Typography variant="caption" color="text.secondary">
                                    (Last updated: {new Date(offWaitTimeProperty.value.lastChangedAt).toLocaleString()})
                                </Typography>
                            )}
                        </Box>
                    </Box>
                )}

                {capability.actions && capability.actions.length > 0 && (
                    <Box>
                        <Typography variant="body2" color="text.secondary">
                            Available Actions:
                        </Typography>
                        <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1, mt: 0.5 }}>
                            {capability.actions.map((action) => (
                                <Chip
                                    key={action}
                                    label={action}
                                    size="small"
                                    variant="outlined"
                                />
                            ))}
                        </Box>
                    </Box>
                )}
            </Stack>
        </Box>
    );
};

// Widget for dashboard cards
export const OnOffWidget: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const dispatch = useDispatch();
    const [isLoading, setIsLoading] = useState(false);
    const [isOn, setIsOn] = useState(() => {
        // Initialize from the actual capability data, just like OnOffDetails
        const onOffProperty = capability?.properties?.find(p => p.name === 'OnOff');
        const initialValue = onOffProperty?.value?.propertyValue === true;
        
        
        // Trust the cloud data - if everything else works, initialization should too
        return initialValue;
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0);
    const [readStateInitialized, setReadStateInitialized] = useState(false); // Track if ReadState was sent

    // REMOVED: ReadState useEffect was causing infinite loops and is disabled anyway

    // Update local state when capability changes, but respect optimistic updates
    useEffect(() => {
        const now = Date.now();
        const timeSinceUserAction = now - lastUserAction;
        
        // Only proceed if capability exists and has the required property
        const onOffProperty = capability?.properties?.find(p => p.name === 'OnOff');
        if (capability && onOffProperty?.value?.propertyValue !== undefined) {
            const newValue = onOffProperty.value.propertyValue === true;
            const lastChangedAt = onOffProperty.value.lastChangedAt;
            
            console.log('OnOffWidget - Capability updated:', {
                deviceId,
                timeSinceUserAction,
                currentIsOn: isOn,
                newValue,
                lastChangedAt,
                timestamp: new Date().toISOString()
            });
            
            // Hybrid approach: timestamp comparison + grace period fallback for clock sync issues
            const mostRecentCommandTime = deviceStatePollingService.getLastCommandTimestamp(deviceId);
            
            if (!mostRecentCommandTime) {
                // Initial load - no commands sent yet, trust the device state
                setIsOn(newValue);
                console.log('OnOffWidget - Initial load, accepting device state:', {
                    deviceId,
                    newValue,
                    timestamp: new Date().toISOString()
                });
            } else if (lastChangedAt) {
                const deviceStateTimestamp = new Date(lastChangedAt).getTime();
                const timeSinceCommand = Date.now() - mostRecentCommandTime;
                
                // Primary check: timestamp comparison + minimum threshold
                // Only accept device state if it's newer than command AND at least 5 seconds have passed
                if (deviceStateTimestamp > mostRecentCommandTime && timeSinceCommand > 5000) {
                    // Device state is newer than our most recent command AND enough time has passed
                    setIsOn(newValue);
                    console.log('OnOffWidget - Device state newer than command + threshold met, accepting:', {
                        deviceId,
                        deviceStateTimestamp: lastChangedAt,
                        mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        newValue,
                        method: 'timestamp-comparison-with-threshold'
                    });
                }
                // Check if timestamp is newer but we're still within the 5-second threshold
                else if (deviceStateTimestamp > mostRecentCommandTime && timeSinceCommand <= 5000) {
                    console.log('OnOffWidget - Device state newer but within 5s threshold, keeping optimistic state:', {
                        deviceId,
                        deviceStateTimestamp: lastChangedAt,
                        mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        keepingState: isOn,
                        deviceWants: newValue,
                        method: 'threshold-protection'
                    });
                }
                // Fallback check: grace period (works when clocks aren't synced)
                else if (timeSinceCommand > 15000) {
                    // It's been more than 8 seconds since command - accept device state as final truth
                    setIsOn(newValue);
                    console.log('OnOffWidget - Grace period expired, accepting device state:', {
                        deviceId,
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        deviceStateTimestamp: lastChangedAt,
                        mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                        newValue,
                        method: 'grace-period-fallback'
                    });
                } 
                else {
                    console.log('OnOffWidget - Keeping optimistic state (timestamp older + within grace period):', {
                        deviceId,
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        deviceStateTimestamp: lastChangedAt,
                        mostRecentCommandTime: new Date(mostRecentCommandTime).toISOString(),
                        keepingState: isOn,
                        deviceWants: newValue,
                        method: 'optimistic-hold'
                    });
                }
            } else {
                // No timestamp available - use grace period only
                const timeSinceCommand = Date.now() - mostRecentCommandTime;
                if (timeSinceCommand > 8000) {
                    setIsOn(newValue);
                    console.log('OnOffWidget - No timestamp, grace period expired, accepting device state:', {
                        deviceId,
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        newValue,
                        method: 'grace-period-only'
                    });
                } else {
                    console.log('OnOffWidget - No timestamp, within grace period, keeping optimistic state:', {
                        deviceId,
                        timeSinceCommand: `${Math.round(timeSinceCommand / 1000)}s`,
                        keepingState: isOn,
                        deviceWants: newValue,
                        method: 'optimistic-hold-no-timestamp'
                    });
                }
            }
        }
    }, [capability?.properties?.find(p => p.name === 'OnOff')?.value?.propertyValue, deviceId, lastUserAction]);

    const handleToggle = async () => {
        if (isLoading) return;
        
        // Store original state for potential revert
        const originalState = isOn;
        
        try {
            // Record user action timestamp to prevent polling interference
            const commandTimestamp = Date.now();
            setLastUserAction(commandTimestamp);
            
            // Record toggle command for sequence-based state tracking
            deviceStatePollingService.recordToggle(deviceId, originalState, !originalState, commandTimestamp);
            
            // OPTIMISTIC UPDATE: Immediately flip the UI
            setIsOn(!isOn);
            console.log('OnOffWidget - Optimistic UI update:', {
                deviceId,
                from: originalState,
                to: !originalState,
                timestamp: new Date().toISOString()
            });
            
            setIsLoading(true);
            
            // Get the device details to get the endpoint ID from capability report
            const device = await deviceService.getDevice(deviceId);
            const endpointId = device.CapabilityReport.endpoints[0]?.id;
            
            if (!endpointId) {
                console.error('No endpoint ID found in capability report for device:', deviceId);
                // Revert optimistic update on error
                setIsOn(originalState);
                return;
            }

            // Send the appropriate command based on ORIGINAL state (before optimistic update)
            const command = originalState ? 'Off' : 'On';
            
            // Log command timing for comparison with GetState responses
            console.log(`COMMAND_SENT: Device ${deviceId}, Command ${command}, Timestamp ${commandTimestamp}, ISO ${new Date(commandTimestamp).toISOString()}`);
            
            await deviceService.executeAction(deviceId, capability.id, command, endpointId);
            
            console.log('OnOffWidget - Command sent successfully:', {
                deviceId,
                command,
                endpointId,
                timestamp: new Date().toISOString()
            });

            // The DeviceStatePollingService will automatically update the state
            // No need to manually poll here

        } catch (error) {
            console.error('Error toggling device:', error);
            // Revert optimistic update on error, but extend grace period
            // to prevent immediate polling interference
            setIsOn(originalState);
            setLastUserAction(Date.now()); // Reset grace period to prevent polling flicker
            console.log('OnOffWidget - Reverted UI due to error:', {
                deviceId,
                revertedTo: originalState,
                error: error instanceof Error ? error.message : 'Unknown error',
                timestamp: new Date().toISOString()
            });
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <Box sx={{ 
            p: 2, 
            borderRadius: 2,
            bgcolor: 'background.paper',
            position: 'relative',
            minHeight: 100,
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
            gap: 2
        }}>
            <Box sx={getIconContainerStyles(isOn)}>
                {isOn ? (
                    <Icons.Lightbulb sx={getIconStyles(isOn)} />
                ) : (
                    <Icons.LightbulbOutlined sx={getIconStyles(isOn)} />
                )}
            </Box>
            
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                <Typography variant="body1" color="text.secondary">
                    {isOn ? 'On' : 'Off'}
                </Typography>
                <IOSSwitch
                    checked={isOn}
                    onChange={handleToggle}
                    disabled={isLoading}
                />
            </Box>

            {isLoading && (
                <CircularProgress
                    size={24}
                    sx={{
                        position: 'absolute',
                        top: '50%',
                        left: '50%',
                        marginTop: '-12px',
                        marginLeft: '-12px',
                    }}
                />
            )}
        </Box>
    );
};

// Cluster implementation
const OnOffCluster: ClusterImplementation = Object.assign(
    // The default component render
    ({ capability, deviceId }: ClusterProps) => {
        return <OnOffDetails capability={capability} deviceId={deviceId} />;
    },
    // The interface methods
    {
        renderDetails: ({ capability, deviceId }: ClusterProps) => {
            return <OnOffDetails capability={capability} deviceId={deviceId} />;
        },
        renderWidget: ({ capability, deviceId }: ClusterProps) => {
            return <OnOffWidget capability={capability} deviceId={deviceId} />;
        }
    }
);

export default OnOffCluster;