import React, { useState, useEffect, useCallback } from 'react';
import {
    Box,
    Typography,
    Slider,
    CircularProgress,
    Paper,
    Grid,
    Switch,
    Button
} from '@mui/material';
import { deviceService } from '../services/deviceService';
import { Capability } from '../types/deviceTypes';
import { ClusterProps } from './index';
import { debugLogger } from '../utils/debugLogger';

interface LevelControlCapability extends Capability {
    properties: {
        name: string;
        value?: {
            propertyValue: any;
            lastChangedAt: string;
        };
    }[];
}

// Widget for dashboard cards
export const LevelControlWidget: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const [controlState, setControlState] = useState({
        level: 50
    });
    const [loading, setLoading] = useState(false);
    const [currentState, setCurrentState] = useState({
        level: 0,
        lastUpdated: ''
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0);
    const [debounceTimer, setDebounceTimer] = useState<NodeJS.Timeout | null>(null);

    // Simplified state update - since only the app controls brightness, trust device state after initialization
    useEffect(() => {
        if (!capability?.properties) return;
        
        const now = Date.now();
        const timeSinceUserAction = now - lastUserAction;
        
        const currentLevelProp = capability.properties.find(p => p.name === 'CurrentLevel');
        if (currentLevelProp?.value?.propertyValue !== undefined) {
            // Level values are 0-254, convert to percentage
            const rawLevel = Number(currentLevelProp.value.propertyValue);
            const percentage = Math.round((rawLevel / 254) * 100);
            
            // Simple approach: only avoid updates during active user interaction (brief grace period)
            if (timeSinceUserAction > 2000 || timeSinceUserAction === 0) {
                setCurrentState({
                    level: percentage,
                    lastUpdated: currentLevelProp.value?.lastChangedAt || ''
                });
                // Also update control state if user isn't actively dragging
                if (timeSinceUserAction === 0) {
                    setControlState({ level: percentage });
                }
                debugLogger.log('LevelControlWidget - Updated from device state:', {
                    deviceId,
                    level: percentage,
                    timeSinceUserAction: `${Math.round(timeSinceUserAction / 1000)}s`
                });
            } else {
                debugLogger.log('LevelControlWidget - Ignoring update during user interaction:', {
                    deviceId,
                    timeSinceUserAction: `${Math.round(timeSinceUserAction / 1000)}s`
                });
            }
        }
    }, [capability?.properties?.find(p => p.name === 'CurrentLevel')?.value?.propertyValue, lastUserAction, deviceId]);

    const applyLevelRealtime = async (level: number) => {
        if (!deviceId || !capability) return;
        
        try {
            console.log('LevelControlWidget - Real-time brightness update:', {
                deviceId,
                level,
                timestamp: new Date().toISOString()
            });
            
            // Convert percentage to 0-254 range
            const rawLevel = Math.round((level / 100) * 254);
            
            // Get the endpoint ID
            const deviceState = await deviceService.getDeviceState(deviceId);
            const endpointId = deviceState?.Endpoints?.[0]?.endpointId;
            
            if (!endpointId) {
                console.error('No endpoint ID found for device:', deviceId);
                return;
            }

            // Execute the level control action with smooth transition (fire-and-forget)
            deviceService.executeAction(
                deviceId,
                capability.id,
                'MoveToLevel',
                endpointId,
                { Level: rawLevel, TransitionTime: 5 } // Smooth 0.5-second transition
            ).then(() => {
                console.log('LevelControlWidget - Real-time command sent successfully:', {
                    deviceId,
                    rawLevel,
                    endpointId
                });
            }).catch((error) => {
                console.error('Error setting level in real-time:', error);
            });
            
        } catch (error) {
            console.error('Error setting level in real-time:', error);
        }
    };

    const handleLevelChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState({
            level: value
        });
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        // Clear existing debounce timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }
        
        // Set new debounce timer for real-time updates
        const timer = setTimeout(() => {
            applyLevelRealtime(value);
        }, 100); // 100ms debounce for ultra-responsive control
        
        setDebounceTimer(timer);
    };

    // Conditional rendering at the END, after all hooks are called
    if (!capability) {
        return null;
    }

    return (
        <Box sx={{ position: 'relative' }}>
            <Paper sx={{ p: 2, borderRadius: 2, bgcolor: 'background.paper' }}>
                <Typography variant="subtitle2" color="text.secondary" gutterBottom>
                    Brightness
                </Typography>
                <Box sx={{ px: 1 }}>
                    <Slider
                        value={controlState.level}
                        onChange={handleLevelChange}
                        disabled={loading}
                        aria-label="Brightness"
                        valueLabelDisplay="auto"
                        valueLabelFormat={(value) => `${value}%`}
                    />
                </Box>
                <Typography variant="body2" align="center" gutterBottom>
                    {controlState.level}%
                </Typography>
                
                {loading && (
                    <CircularProgress
                        size={24}
                        sx={{
                            position: 'absolute',
                            top: '50%',
                            left: '50%',
                            marginTop: '-12px',
                            marginLeft: '-12px',
                        }}
                    />
                )}
            </Paper>
        </Box>
    );
};

// Details view component
const LevelControlDetails: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const [controlState, setControlState] = useState({
        level: 50
    });
    const [loading, setLoading] = useState(false);
    const [levelDetails, setLevelDetails] = useState({
        minLevel: 0,
        maxLevel: 254,
        onOffTransitionTime: 0,
        remainingTime: 0
    });
    const [currentState, setCurrentState] = useState({
        level: 0,
        lastUpdated: ''
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0);
    const [debounceTimer, setDebounceTimer] = useState<NodeJS.Timeout | null>(null);

    // Find all relevant properties with optimistic update protection
    useEffect(() => {
        if (!capability || !capability.properties) return;
        
        const now = Date.now();
        const timeSinceUserAction = now - lastUserAction;
        
        const currentLevelProp = capability.properties.find(p => p.name === 'CurrentLevel');
        const minLevelProp = capability.properties.find(p => p.name === 'MinLevel');
        const maxLevelProp = capability.properties.find(p => p.name === 'MaxLevel');
        const remainingTimeProp = capability.properties.find(p => p.name === 'RemainingTime');
        const onOffTransitionTimeProp = capability.properties.find(p => p.name === 'OnOffTransitionTime');

        console.log('LevelControlDetails - Raw capability data:', {
            currentLevelProp: currentLevelProp?.value?.propertyValue,
            timeSinceUserAction,
            lastUserAction
        });

        if (currentLevelProp?.value?.propertyValue !== undefined) {
            const rawLevel = Number(currentLevelProp.value.propertyValue);
            const percentage = Math.round((rawLevel / 254) * 100);
            
            console.log('LevelControlDetails - Level conversion:', {
                rawLevel,
                percentage,
                timeSinceUserAction
            });
            
            // Only update from polling if enough time has passed since user action
            if (timeSinceUserAction > 3000 || timeSinceUserAction === 0) {
                setCurrentState({
                    level: percentage,
                    lastUpdated: currentLevelProp.value?.lastChangedAt || ''
                });
                // Also update control state if user isn't actively dragging
                if (timeSinceUserAction === 0) {
                    setControlState({ level: percentage });
                    console.log('LevelControlDetails - Updated control state to:', percentage);
                }
            } else {
                console.log('LevelControlDetails - Ignoring polling update due to recent user action');
            }
        } else {
            console.log('LevelControlDetails - No CurrentLevel property found or undefined value');
        }

        setLevelDetails({
            minLevel: minLevelProp?.value?.propertyValue || 0,
            maxLevel: maxLevelProp?.value?.propertyValue || 254,
            onOffTransitionTime: onOffTransitionTimeProp?.value?.propertyValue || 0,
            remainingTime: remainingTimeProp?.value?.propertyValue || 0
        });
    }, [capability, lastUserAction]);

    const applyLevelCommand = useCallback(async (level: number) => {
        if (!deviceId || !capability) return;
        
        try {
            setLoading(true);
            console.log('LevelControlDetails - Sending brightness command:', {
                deviceId,
                level,
                timestamp: new Date().toISOString()
            });
            
            // Convert percentage to 0-254 range
            const rawLevel = Math.round((level / 100) * 254);
            
            // Get the endpoint ID
            const deviceState = await deviceService.getDeviceState(deviceId);
            const endpointId = deviceState?.Endpoints?.[0]?.endpointId;
            
            if (!endpointId) {
                console.error('No endpoint ID found for device:', deviceId);
                return;
            }

            // Execute the level control action - let the bulb handle smooth transitions
            await deviceService.executeAction(
                deviceId,
                capability.id,
                'MoveToLevel',
                endpointId,
                { Level: rawLevel, TransitionTime: 10 } // 1 second smooth transition
            );
            
            console.log('LevelControlDetails - Brightness command sent successfully:', {
                deviceId,
                rawLevel,
                endpointId
            });
        } catch (error) {
            console.error('Error setting level:', error);
        } finally {
            setLoading(false);
        }
    }, [deviceId, capability]);

    const handleLevelChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState({
            level: value
        });
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        // Clear existing debounce timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }
        
        // Set new debounce timer - only send command when slider stops moving
        const timer = setTimeout(() => {
            applyLevelCommand(value);
        }, 500); // 500ms after slider stops moving
        
        setDebounceTimer(timer);
    };

    // Format timestamp to a readable format
    const formatTimestamp = (timestamp: string) => {
        if (!timestamp) return "Never";
        const date = new Date(timestamp);
        return date.toLocaleTimeString();
    };

    return (
        <Box>
            {/* Main brightness control - matching widget style */}
            <Box sx={{ 
                p: 3, 
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                position: 'relative',
                mb: 3,
                transition: 'all 0.3s ease',
                '&:hover': {
                    boxShadow: 2
                }
            }}>
                <Typography variant="subtitle1" color="text.secondary" gutterBottom align="center">
                    Brightness
                </Typography>
                
                <Box sx={{ textAlign: 'center', mb: 3 }}>
                    <Typography variant="h2" sx={{ 
                        color: 'primary.main',
                        fontWeight: 'bold',
                        mb: 1
                    }}>
                        {controlState.level}%
                    </Typography>
                </Box>
                
                <Box sx={{ px: 1, mb: 2 }}>
                    <Slider
                        value={controlState.level}
                        onChange={handleLevelChange}
                        disabled={loading}
                        aria-label="Brightness"
                        valueLabelDisplay="auto"
                        valueLabelFormat={(value) => `${value}%`}
                        onWheel={(e) => e.preventDefault()} // Prevent scroll wheel interference
                        sx={{
                            '& .MuiSlider-thumb': {
                                width: 20,
                                height: 20,
                            },
                            '& .MuiSlider-track': {
                                height: 6,
                            },
                            '& .MuiSlider-rail': {
                                height: 6,
                            },
                        }}
                    />
                </Box>
                
                {currentState.lastUpdated && (
                    <Typography variant="caption" color="text.secondary" align="center" display="block">
                        Last updated: {formatTimestamp(currentState.lastUpdated)}
                    </Typography>
                )}
                
                {loading && (
                    <CircularProgress
                        size={24}
                        sx={{
                            position: 'absolute',
                            top: 16,
                            right: 16,
                        }}
                    />
                )}
            </Box>
            
            {/* Device details - compact */}
            <Box sx={{ 
                p: 2,
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                mb: 2
            }}>
                <Typography variant="body2" color="text.secondary" gutterBottom>
                    Device Details
                </Typography>
                <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 1 }}>
                    <Typography variant="caption" color="text.secondary">
                        Min Level: {levelDetails.minLevel}
                    </Typography>
                    <Typography variant="caption" color="text.secondary">
                        Max Level: {levelDetails.maxLevel}
                    </Typography>
                    <Typography variant="caption" color="text.secondary">
                        Transition: {levelDetails.onOffTransitionTime}ms
                    </Typography>
                    <Typography variant="caption" color="text.secondary">
                        Remaining: {levelDetails.remainingTime}ms
                    </Typography>
                </Box>
            </Box>
        </Box>
    );
};

export default {
    renderWidget: LevelControlWidget,
    renderDetails: LevelControlDetails
};