import React, { useState, useEffect, useCallback } from 'react';
import {
    Box,
    Typography,
    Slider,
    CircularProgress,
    Paper
} from '@mui/material';
import { deviceService } from '../services/deviceService';
import { Capability } from '../types/deviceTypes';
import { ClusterProps } from './index';
import { debugLogger } from '../utils/debugLogger';
import { deviceStatePollingService } from '../services/deviceStatePollingService';

interface ColorControlCapability extends Capability {
    properties: {
        name: string;
        value?: {
            propertyValue: any;
            lastChangedAt: string;
        };
    }[];
}

// Widget for dashboard cards - includes brightness, hue, and saturation together
export const ColorControlWidget: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    debugLogger.log('ColorControlWidget - Rendering with props:', { capability, deviceId });
    
    const [controlState, setControlState] = useState({
        brightness: 50,
        hue: 180,
        saturation: 80,
        isOn: true
    });
    const [loading, setLoading] = useState(false);
    const [currentState, setCurrentState] = useState({
        brightness: 0,
        hue: 0,
        saturation: 0,
        isOn: false,
        lastUpdated: ''
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0);
    const [debounceTimers, setDebounceTimers] = useState<{[key: string]: NodeJS.Timeout}>({});

    // Optimistic updates approach - only apply device state on initialization
    const [initialized, setInitialized] = useState(false);
    
    useEffect(() => {
        if (!capability?.properties) return;

        const currentHueProp = capability.properties.find((p: any) => p.name === 'CurrentHue');
        const currentSatProp = capability.properties.find((p: any) => p.name === 'CurrentSaturation');
        
        // Only update from device state on initialization (when no user actions have occurred)
        if (!initialized && lastUserAction === 0) {
            const updates: any = {};
            
            if (currentHueProp?.value?.propertyValue !== undefined) {
                const rawHue = Number(currentHueProp.value.propertyValue);
                const degrees = Math.round((rawHue / 254) * 360);
                updates.hue = degrees;
            }
            
            if (currentSatProp?.value?.propertyValue !== undefined) {
                const rawSat = Number(currentSatProp.value.propertyValue);
                const percentage = Math.round((rawSat / 254) * 100);
                updates.saturation = percentage;
            }
            
            if (Object.keys(updates).length > 0) {
                setControlState(prev => ({ ...prev, ...updates }));
                setCurrentState(prev => ({ 
                    ...prev, 
                    ...updates,
                    lastUpdated: currentHueProp?.value?.lastChangedAt || currentSatProp?.value?.lastChangedAt || prev.lastUpdated
                }));
                setInitialized(true);
                debugLogger.log('ColorControlWidget - Initialized from device state:', {
                    deviceId,
                    updates,
                    timestamp: new Date().toISOString()
                });
            }
        } else if (initialized) {
            debugLogger.log('ColorControlWidget - Ignoring device state (using optimistic updates only):', {
                deviceId,
                initialized,
                lastUserAction,
                timestamp: new Date().toISOString()
            });
        }
    }, [capability?.properties?.find(p => p.name === 'CurrentHue')?.value?.propertyValue, capability?.properties?.find(p => p.name === 'CurrentSaturation')?.value?.propertyValue, lastUserAction, deviceId, initialized]);

    // Clear debounce timer helper
    const clearDebounceTimer = (key: string) => {
        if (debounceTimers[key]) {
            clearTimeout(debounceTimers[key]);
            setDebounceTimers(prev => {
                const newTimers = { ...prev };
                delete newTimers[key];
                return newTimers;
            });
        }
    };

    // Set debounce timer helper
    const setDebounceTimer = (key: string, callback: () => void, delay: number) => {
        clearDebounceTimer(key);
        const timer = setTimeout(callback, delay);
        setDebounceTimers(prev => ({ ...prev, [key]: timer }));
    };

    const applyColorCommand = useCallback(async (hue: number, saturation: number) => {
        if (!deviceId || !capability) return;
        
        try {
            setLoading(true);
            
            // Record command timestamp to prevent polling interference (OnOffCluster style)
            const commandTimestamp = Date.now();
            deviceStatePollingService.recordCommand(deviceId, commandTimestamp);
            
            debugLogger.log('ColorControlWidget - Sending color command:', {
                deviceId,
                hue,
                saturation,
                commandTimestamp,
                timestamp: new Date().toISOString()
            });
            
            // Get the endpoint ID
            const deviceState = await deviceService.getDeviceState(deviceId);
            const endpointId = deviceState?.Endpoints?.[0]?.endpointId;
            
            if (!endpointId) {
                debugLogger.error('No endpoint ID found for device:', deviceId);
                return;
            }

            // Convert to raw values (0-254 range)
            const rawHue = Math.round((hue / 360) * 254);
            const rawSat = Math.round((saturation / 100) * 254);

            // Execute the color control action - let the bulb handle smooth transitions
            await deviceService.executeAction(
                deviceId,
                capability.id,
                'MoveToHueAndSaturation',
                endpointId,
                { Hue: rawHue, Saturation: rawSat, TransitionTime: 10 } // 1 second smooth transition
            );
            
            debugLogger.log('ColorControlWidget - Color command sent successfully:', {
                deviceId,
                rawHue,
                rawSat,
                endpointId,
                commandTimestamp
            });
        } catch (error) {
            debugLogger.error('Error setting color:', error);
        } finally {
            setLoading(false);
        }
    }, [deviceId, capability?.id]);

    // This widget only handles color (hue/saturation) control

    const handleHueChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState(prev => ({
            ...prev,
            hue: value
        }));
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        setDebounceTimer('hue', () => {
            applyColorCommand(value, controlState.saturation);
        }, 500);
    };

    const handleSaturationChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState(prev => ({
            ...prev,
            saturation: value
        }));
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        setDebounceTimer('saturation', () => {
            applyColorCommand(controlState.hue, value);
        }, 500);
    };

    // Conditional rendering - always render if capability exists (capabilities are source of truth)
    if (!capability) {
        return (
            <Box sx={{ position: 'relative' }}>
                <Paper sx={{ p: 2, borderRadius: 2, bgcolor: 'error.light' }}>
                    <Typography variant="subtitle2" color="error">
                        Error: No ColorControl capability
                    </Typography>
                </Paper>
            </Box>
        );
    }

    // Color preview for desired state - circular design
    const desiredColorPreview = {
        background: `radial-gradient(circle, hsl(${controlState.hue}, ${controlState.saturation}%, 60%) 0%, hsl(${controlState.hue}, ${controlState.saturation}%, 40%) 70%, hsl(${controlState.hue}, ${controlState.saturation}%, 20%) 100%)`,
        width: '60px',
        height: '60px',
        borderRadius: '50%',
        margin: '0 auto 16px auto',
        boxShadow: `0 4px 12px hsla(${controlState.hue}, ${controlState.saturation}%, 50%, 0.3)`,
        border: '2px solid rgba(255, 255, 255, 0.2)'
    };

    return (
        <Box sx={{ position: 'relative' }}>
            <Paper sx={{ p: 2, borderRadius: 2, bgcolor: 'background.paper' }}>
                <Typography variant="subtitle2" color="text.secondary" gutterBottom>
                    Color Light
                </Typography>

                {/* Single unified color preview that shows both color and on/off state */}
                <Box sx={{
                    ...desiredColorPreview,
                    opacity: controlState.isOn ? 1 : 0.3,
                    filter: controlState.isOn ? 'none' : 'grayscale(50%)',
                    transition: 'all 0.3s ease'
                }} />

                {/* Color controls only - no brightness */}
                <Box sx={{ px: 1 }}>
                    {/* Hue Control */}
                    <Box sx={{ mb: 1.5 }}>
                        <Typography variant="body2" sx={{ fontSize: '0.875rem', mb: 0.5 }}>
                            Hue ({controlState.hue}°)
                        </Typography>
                        <Slider
                            value={controlState.hue}
                            onChange={handleHueChange}
                            disabled={loading}
                            min={0}
                            max={360}
                            aria-label="Hue"
                            size="small"
                        />
                    </Box>

                    {/* Saturation Control */}
                    <Box sx={{ mb: 1 }}>
                        <Typography variant="body2" sx={{ fontSize: '0.875rem', mb: 0.5 }}>
                            Saturation ({controlState.saturation}%)
                        </Typography>
                        <Slider
                            value={controlState.saturation}
                            onChange={handleSaturationChange}
                            disabled={loading}
                            aria-label="Saturation"
                            size="small"
                        />
                    </Box>
                </Box>

                {loading && (
                    <CircularProgress
                        size={20}
                        sx={{
                            position: 'absolute',
                            top: 16,
                            right: 16,
                        }}
                    />
                )}
            </Paper>
        </Box>
    );
};

// Details view component
const ColorControlDetails: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const [controlState, setControlState] = useState({
        hue: 180,
        saturation: 80
    });
    const [loading, setLoading] = useState(false);
    const [currentState, setCurrentState] = useState({
        hue: 0,
        saturation: 0,
        lastUpdated: ''
    });
    const [lastUserAction, setLastUserAction] = useState<number>(0);
    const [debounceTimer, setDebounceTimer] = useState<NodeJS.Timeout | null>(null);

    // Optimistic updates approach - only apply device state on initialization
    const [initialized, setInitialized] = useState(false);
    
    useEffect(() => {
        if (!capability?.properties) return;
        
        const currentHueProp = capability.properties.find(p => p.name === 'CurrentHue');
        const currentSatProp = capability.properties.find(p => p.name === 'CurrentSaturation');

        // Only update from device state on initialization (when no user actions have occurred)
        if (!initialized && lastUserAction === 0) {
            const updates: any = {};
            
            if (currentHueProp?.value?.propertyValue !== undefined) {
                const rawHue = Number(currentHueProp.value.propertyValue);
                const degrees = Math.round((rawHue / 254) * 360);
                updates.hue = degrees;
            }
            
            if (currentSatProp?.value?.propertyValue !== undefined) {
                const rawSat = Number(currentSatProp.value.propertyValue);
                const percentage = Math.round((rawSat / 254) * 100);
                updates.saturation = percentage;
            }
            
            if (Object.keys(updates).length > 0) {
                setControlState(prev => ({ ...prev, ...updates }));
                setCurrentState(prev => ({ 
                    ...prev, 
                    ...updates,
                    lastUpdated: currentHueProp?.value?.lastChangedAt || currentSatProp?.value?.lastChangedAt || prev.lastUpdated
                }));
                setInitialized(true);
                debugLogger.log('ColorControlDetails - Initialized from device state:', {
                    deviceId,
                    updates,
                    timestamp: new Date().toISOString()
                });
            }
        } else if (initialized) {
            debugLogger.log('ColorControlDetails - Ignoring device state (using optimistic updates only):', {
                deviceId,
                initialized,
                lastUserAction,
                timestamp: new Date().toISOString()
            });
        }
    }, [capability?.properties?.find(p => p.name === 'CurrentHue')?.value?.propertyValue, capability?.properties?.find(p => p.name === 'CurrentSaturation')?.value?.propertyValue, lastUserAction, deviceId, initialized]);

    const applyColorCommand = useCallback(async (hue: number, saturation: number) => {
        if (!deviceId || !capability?.id) return;
        
        try {
            setLoading(true);
            
            // Record command timestamp to prevent polling interference (OnOffCluster style)
            const commandTimestamp = Date.now();
            deviceStatePollingService.recordCommand(deviceId, commandTimestamp);
            
            debugLogger.log('ColorControlDetails - Sending color command:', {
                deviceId,
                hue,
                saturation,
                commandTimestamp,
                timestamp: new Date().toISOString()
            });
            
            // Get the endpoint ID
            const deviceState = await deviceService.getDeviceState(deviceId);
            const endpointId = deviceState?.Endpoints?.[0]?.endpointId;
            
            if (!endpointId) {
                debugLogger.error('No endpoint ID found for device:', deviceId);
                return;
            }

            // Convert to raw values (0-254 range)
            const rawHue = Math.round((hue / 360) * 254);
            const rawSat = Math.round((saturation / 100) * 254);

            // Execute the color control action
            await deviceService.executeAction(
                deviceId,
                capability.id,
                'MoveToHueAndSaturation',
                endpointId,
                { Hue: rawHue, Saturation: rawSat, TransitionTime: 5 } // 0.5 second transition
            );
            
            debugLogger.log('ColorControlDetails - Color command sent successfully:', {
                deviceId,
                rawHue,
                rawSat,
                endpointId,
                commandTimestamp
            });
        } catch (error) {
            debugLogger.error('Error setting color:', error);
        } finally {
            setLoading(false);
        }
    }, [deviceId, capability?.id]);

    const handleHueChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState(prev => ({
            ...prev,
            hue: value
        }));
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        // Clear existing debounce timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }
        
        // Set new debounce timer - only send command when slider stops moving
        const timer = setTimeout(() => {
            applyColorCommand(value, controlState.saturation);
        }, 500); // 500ms after slider stops moving
        
        setDebounceTimer(timer);
    };

    const handleSaturationChange = (event: Event, newValue: number | number[]) => {
        const value = newValue as number;
        setControlState(prev => ({
            ...prev,
            saturation: value
        }));
        
        // Record user action to prevent polling interference
        setLastUserAction(Date.now());
        
        // Clear existing debounce timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }
        
        // Set new debounce timer - only send command when slider stops moving
        const timer = setTimeout(() => {
            applyColorCommand(controlState.hue, value);
        }, 500); // 500ms after slider stops moving
        
        setDebounceTimer(timer);
    };

    // Format timestamp to a readable format
    const formatTimestamp = (timestamp: string) => {
        if (!timestamp) return "Never";
        const date = new Date(timestamp);
        return date.toLocaleTimeString();
    };

    // Color preview matching widget style
    const colorPreview = {
        background: `radial-gradient(circle, hsl(${controlState.hue}, ${controlState.saturation}%, 60%) 0%, hsl(${controlState.hue}, ${controlState.saturation}%, 40%) 70%, hsl(${controlState.hue}, ${controlState.saturation}%, 20%) 100%)`,
        width: '80px',
        height: '80px',
        borderRadius: '50%',
        margin: '0 auto 24px auto',
        boxShadow: `0 6px 20px hsla(${controlState.hue}, ${controlState.saturation}%, 50%, 0.4)`,
        border: '3px solid rgba(255, 255, 255, 0.2)'
    };

    return (
        <Box>
            {/* Main control section - matching widget style */}
            <Box sx={{ 
                p: 3,
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                position: 'relative',
                mb: 3,
                transition: 'all 0.3s ease',
                '&:hover': {
                    boxShadow: 2
                }
            }}>
                <Typography variant="subtitle1" color="text.secondary" gutterBottom align="center">
                    Color Light
                </Typography>

                {/* Color preview - larger version of widget */}
                <Box sx={colorPreview} />

                {/* Color controls - matching widget layout */}
                <Box sx={{ px: 1 }}>
                    {/* Hue Control */}
                    <Box sx={{ mb: 2 }}>
                        <Typography variant="body2" sx={{ fontSize: '0.875rem', mb: 1 }}>
                            Hue ({controlState.hue}°)
                        </Typography>
                        <Slider
                            value={controlState.hue}
                            onChange={handleHueChange}
                            disabled={loading}
                            min={0}
                            max={360}
                            aria-label="Hue"
                            onWheel={(e) => e.preventDefault()} // Prevent scroll wheel interference
                            sx={{
                                '& .MuiSlider-track': {
                                    height: 6,
                                    backgroundColor: 'primary.main'
                                },
                                '& .MuiSlider-rail': {
                                    height: 6,
                                },
                                '& .MuiSlider-thumb': {
                                    width: 20,
                                    height: 20,
                                    backgroundColor: 'primary.main',
                                    border: '2px solid white',
                                    boxShadow: '0 2px 8px rgba(0,0,0,0.3)'
                                }
                            }}
                        />
                    </Box>

                    {/* Saturation Control */}
                    <Box sx={{ mb: 1 }}>
                        <Typography variant="body2" sx={{ fontSize: '0.875rem', mb: 1 }}>
                            Saturation ({controlState.saturation}%)
                        </Typography>
                        <Slider
                            value={controlState.saturation}
                            onChange={handleSaturationChange}
                            disabled={loading}
                            aria-label="Saturation"
                            onWheel={(e) => e.preventDefault()} // Prevent scroll wheel interference
                            sx={{
                                '& .MuiSlider-track': {
                                    height: 6,
                                    backgroundColor: 'primary.main'
                                },
                                '& .MuiSlider-rail': {
                                    height: 6,
                                },
                                '& .MuiSlider-thumb': {
                                    width: 20,
                                    height: 20,
                                    backgroundColor: 'primary.main',
                                    border: '2px solid white',
                                    boxShadow: '0 2px 8px rgba(0,0,0,0.3)'
                                }
                            }}
                        />
                    </Box>
                </Box>

                {/* Current state info - subtle */}
                {currentState.lastUpdated && (
                    <Typography variant="caption" color="text.secondary" align="center" display="block" sx={{ mt: 2 }}>
                        Last updated: {formatTimestamp(currentState.lastUpdated)}
                    </Typography>
                )}

                {loading && (
                    <CircularProgress
                        size={24}
                        sx={{
                            position: 'absolute',
                            top: 16,
                            right: 16,
                        }}
                    />
                )}
            </Box>
        </Box>
    );
};

export default {
    renderWidget: ColorControlWidget,
    renderDetails: ColorControlDetails
};