import React from 'react';
import {
    Box,
    Typography,
    Paper,
    Stack,
    Chip,
    Card,
    CardContent
} from '@mui/material';
import * as Icons from '@mui/icons-material';
import { Capability } from '../types/deviceTypes';
import { ClusterImplementation, ClusterProps } from '../types/clusterTypes';

// Helper function to format boolean state
const formatBooleanState = (value: any): string => {
    if (value === undefined || value === null) return 'Unknown';
    return value ? 'Open/Alarmed' : 'Closed/Not Alarmed';
};

// Helper function to get state color and icon
const getStateInfo = (value: boolean) => {
    if (value) {
        return {
            color: '#F44336', // Red for alarmed
            icon: Icons.CheckCircle,
            label: 'Open/Alarmed'
        };
    } else {
        return {
            color: '#4CAF50', // Green for not alarmed
            icon: Icons.RadioButtonUnchecked,
            label: 'Closed/Not Alarmed'
        };
    }
};

// Details view component
const BooleanStateDetails: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const stateValueProperty = capability.properties.find(p => p.name === 'StateValue');
    const currentState = stateValueProperty?.value?.propertyValue;
    const stateInfo = getStateInfo(currentState === true);

    const StateIcon = stateInfo.icon;

    return (
        <Box>
            <Box sx={{ 
                display: 'flex', 
                alignItems: 'center', 
                gap: 3,
                mb: 2,
                p: 3,
                borderRadius: 2,
                bgcolor: 'background.paper',
                boxShadow: 1,
                position: 'relative',
                transition: 'all 0.3s ease',
                '&:hover': {
                    boxShadow: 2
                }
            }}>
                <Box sx={{
                    position: 'relative',
                    width: 64,
                    height: 64,
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    borderRadius: '50%',
                    bgcolor: `${stateInfo.color}20`,
                    border: `2px solid ${stateInfo.color}`,
                    transition: 'all 0.3s ease'
                }}>
                    <StateIcon sx={{ 
                        fontSize: 40, 
                        color: stateInfo.color,
                        transition: 'all 0.3s ease'
                    }} />
                </Box>

                <Box sx={{ flexGrow: 1 }}>
                    <Typography variant="h4" sx={{ 
                        color: stateInfo.color,
                        fontWeight: 'bold',
                        mb: 1
                    }}>
                        {stateInfo.label}
                    </Typography>
                    <Typography variant="h6" color="text.secondary">
                        {capability.name}
                    </Typography>
                    {stateValueProperty?.value?.lastChangedAt && (
                        <Typography variant="caption" color="text.secondary">
                            Last updated: {new Date(stateValueProperty.value.lastChangedAt).toLocaleString()}
                        </Typography>
                    )}
                </Box>
            </Box>

            <Stack spacing={2} sx={{ mt: 3 }}>
                <Card variant="outlined">
                    <CardContent sx={{ py: 2 }}>
                        <Typography variant="body2" color="text.secondary" gutterBottom>
                            Current State
                        </Typography>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <Chip 
                                label={formatBooleanState(currentState)}
                                color={currentState ? 'error' : 'success'}
                                variant={currentState ? 'filled' : 'filled'}
                            />
                            {stateValueProperty?.value?.lastChangedAt && (
                                <Typography variant="caption" color="text.secondary">
                                    (Last changed: {new Date(stateValueProperty.value.lastChangedAt).toLocaleString()})
                                </Typography>
                            )}
                        </Box>
                    </CardContent>
                </Card>

                {capability.actions && capability.actions.length > 0 && (
                    <Box>
                        <Typography variant="body2" color="text.secondary">
                            Available Actions:
                        </Typography>
                        <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1, mt: 0.5 }}>
                            {capability.actions.map((action) => (
                                <Chip
                                    key={action}
                                    label={action}
                                    size="small"
                                    variant="outlined"
                                />
                            ))}
                        </Box>
                    </Box>
                )}
            </Stack>
        </Box>
    );
};

// Widget for dashboard cards
export const BooleanStateWidget: React.FC<ClusterProps> = ({ capability, deviceId }) => {
    const stateValueProperty = capability.properties.find(p => p.name === 'StateValue');
    const currentState = stateValueProperty?.value?.propertyValue;
    const stateInfo = getStateInfo(currentState === true);

    const StateIcon = stateInfo.icon;

    return (
        <Box sx={{ 
            p: 2, 
            borderRadius: 2,
            bgcolor: 'background.paper',
            position: 'relative',
            minHeight: 100,
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
            gap: 2
        }}>
            <Box sx={{
                position: 'relative',
                width: 48,
                height: 48,
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                borderRadius: '50%',
                bgcolor: `${stateInfo.color}20`,
                border: `2px solid ${stateInfo.color}`,
                transition: 'all 0.3s ease'
            }}>
                <StateIcon sx={{ 
                    fontSize: 32, 
                    color: stateInfo.color,
                    transition: 'all 0.3s ease'
                }} />
            </Box>
            
            <Box sx={{ textAlign: 'center' }}>
                <Typography variant="h6" sx={{ 
                    color: stateInfo.color,
                    fontWeight: 'bold'
                }}>
                    {stateInfo.label}
                </Typography>
                <Typography variant="body2" color="text.secondary">
                    State
                </Typography>
            </Box>
        </Box>
    );
};

// Cluster implementation
const BooleanStateCluster: ClusterImplementation = Object.assign(
    // The default component render
    ({ capability, deviceId }: ClusterProps) => {
        return <BooleanStateDetails capability={capability} deviceId={deviceId} />;
    },
    // The interface methods
    {
        renderDetails: ({ capability, deviceId }: ClusterProps) => {
            return <BooleanStateDetails capability={capability} deviceId={deviceId} />;
        },
        renderWidget: ({ capability, deviceId }: ClusterProps) => {
            return <BooleanStateWidget capability={capability} deviceId={deviceId} />;
        }
    }
);

export default BooleanStateCluster;
