import React, { useEffect, useState } from 'react';
import { BrowserRouter as Router, Routes, Route } from 'react-router-dom';
import {
    ThemeProvider,
    CssBaseline,
    Container,
    Box,
    CircularProgress,
    Typography
} from '@mui/material';
import { useDispatch, useSelector } from 'react-redux';
import { Provider } from 'react-redux';

// Store and services
import { store } from './store/store';
import theme from './theme';
import { initializeServices } from './services/initializeServices';
import { useDeviceMonitoring } from './hooks/useDeviceMonitoring';

// Constants
import { ERROR_MESSAGES, LOADING_MESSAGES, ROUTES } from './constants';

// Redux selectors and actions
import {
    selectHubLoading,
    selectHubError,
    fetchAndAddAllDevices
} from './store/hubSlice';
import { AppDispatch } from './store/store';

// Components
import Dashboard from './pages/Dashboard';
import DeviceDetail from './components/devices/DeviceDetail';
import Header from './components/layout/Header';
// Removed GlobalNamingDialog - naming is now handled during onboarding
import SimpleOnboarding from './pages/SimpleOnboarding';
import EnvironmentTest from './pages/EnvironmentTest';

function AppContent() {
    const dispatch = useDispatch<AppDispatch>();
    const isLoading = useSelector(selectHubLoading);
    const error = useSelector(selectHubError);
    const [servicesInitialized, setServicesInitialized] = useState(false);
    const [initError, setInitError] = useState<string | null>(null);


    // Initialize services and load devices
    useEffect(() => {
        let mounted = true;

        const initializeApp = async () => {
            try {
                console.log('Starting services initialization...');
                const success = await initializeServices();
                
                if (!success) {
                    throw new Error(ERROR_MESSAGES.INITIALIZATION_FAILED);
                }

                if (mounted) {
                    setServicesInitialized(true);
                    console.log('Services initialized, loading devices...');
                    dispatch(fetchAndAddAllDevices());
                }
            } catch (error) {
                console.error('Error initializing services:', error);
                if (mounted) {
                    setInitError(error instanceof Error ? error.message : ERROR_MESSAGES.UNKNOWN_ERROR);
                }
            }
        };

        initializeApp();

        return () => {
            mounted = false;
        };
    }, [dispatch]);

    // Show initialization loading state
    if (!servicesInitialized && !initError) {
        return (
            <Box sx={{ 
                display: 'flex', 
                flexDirection: 'column', 
                alignItems: 'center', 
                justifyContent: 'center', 
                minHeight: '100vh' 
            }}>
                <CircularProgress />
                <Typography sx={{ mt: 2 }}>{LOADING_MESSAGES.INITIALIZING}</Typography>
            </Box>
        );
    }

    // Show initialization error
    if (initError) {
        return (
            <Box sx={{ 
                display: 'flex', 
                flexDirection: 'column', 
                alignItems: 'center', 
                justifyContent: 'center', 
                minHeight: '100vh',
                textAlign: 'center'
            }}>
                <Typography color="error" variant="h6" sx={{ mb: 2 }}>
                    {ERROR_MESSAGES.INITIALIZATION_FAILED}
                </Typography>
                <Typography color="error">
                    {initError}
                </Typography>
            </Box>
        );
    }

    return (
        <Box sx={{
            minHeight: '100vh',
            backgroundColor: (theme) => theme.palette.background.default
        }}>
            <Header />
            
            {isLoading && (
                <Box sx={{ 
                    display: 'flex', 
                    flexDirection: 'column', 
                    alignItems: 'center', 
                    mt: 2 
                }}>
                    <CircularProgress />
                    <Typography sx={{ mt: 1 }}>{LOADING_MESSAGES.LOADING_DEVICES}</Typography>
                </Box>
            )}
            
            {error && (
                <Box sx={{ 
                    textAlign: 'center', 
                    mt: 2, 
                    mx: 2 
                }}>
                    <Typography color="error">
                        Error loading devices: {error}
                    </Typography>
                </Box>
            )}
            
            <Container maxWidth="lg" sx={{ pt: 4, pb: 4 }}>
                <Routes>
                    <Route path={ROUTES.HOME} element={<Dashboard />} />
                    <Route path={ROUTES.SIMPLE_ONBOARD} element={<SimpleOnboarding />} />
                    <Route path={ROUTES.DEVICE_DETAIL} element={<DeviceDetail />} />
                    <Route path={ROUTES.ENV_TEST} element={<EnvironmentTest />} />
                </Routes>
            </Container>
            
            {/* Removed GlobalNamingDialog - naming is now handled during onboarding */}
        </Box>
    );
}

function App() {
    return (
        <Provider store={store}>
            <ThemeProvider theme={theme}>
                <CssBaseline />
                <Router>
                    <AppContent />
                </Router>
            </ThemeProvider>
        </Provider>
    );
}

export default App;
